<?php

namespace Atrato\PayLater\Model;

use Atrato\PayLater\Api\AtratoOrdersInterface;
use Atrato\PayLater\Helper\Data;
use Atrato\PayLater\Model\Config\Settings;
use Magento\Framework\Controller\Result\JsonFactory;

use Magento\Framework\Webapi\Rest\Request;
use Magento\Sales\Api\OrderRepositoryInterface;
use Magento\Sales\Api\OrderStatusHistoryRepositoryInterface;
use Magento\Sales\Model\Order\Status\HistoryFactory;
use Magento\Store\Model\StoreManagerInterface;


class AtratoOrders implements AtratoOrdersInterface
{
    protected $request;
    protected $orderRepository;
    protected $resultJsonFactory;
    protected $atratoConfig;
    protected $helperData;
    protected $storeManager;
    private $historyRepository;
    private $historyFactory;

    /**
     * @param Request $request
     * @param HistoryFactory $historyFactory
     * @param OrderStatusHistoryRepositoryInterface $historyRepository
     * @param OrderRepositoryInterface $orderRepository
     * @param JsonFactory $resultJsonFactory
     * @param Settings $atratoConfig
     * @param Data $helperData
     * @param StoreManagerInterface $storeManager
     */
    public function __construct(
        Request $request,
        HistoryFactory $historyFactory,
        OrderStatusHistoryRepositoryInterface $historyRepository,
        OrderRepositoryInterface $orderRepository,
        JsonFactory $resultJsonFactory,
        Settings $atratoConfig,
        Data $helperData,
        StoreManagerInterface $storeManager
    ) {
        $this->request = $request;
        $this->orderRepository = $orderRepository;
        $this->resultJsonFactory = $resultJsonFactory;
        $this->atratoConfig = $atratoConfig;
        $this->helperData = $helperData;
        $this->storeManager = $storeManager;
        $this->historyRepository = $historyRepository;
        $this->historyFactory = $historyFactory;
    }

    /**
     * Delete an order when given an id
     * @param string $id
     * @return string
     */
    public function deleteOrder(string $id): string
    {
        // Get the header from the request
        $authenticated = $this->AuthValidation();
        // Validate that they exist
        if ($authenticated) {
            // Validate token to the backend ???
            // Get the order
            $order = $this->orderRepository->get($id);
            $payment_method = $order->getPayment()->getMethod();
            if ($payment_method === "atrato_paylater") {
                $this->orderRepository->delete($order);
                return json_encode(
                    array(
                        'code' => 200,
                        'message' => "Delete order:{$id}  successful",
                    )
                );
            }
            return json_encode(
                array(
                    'code' => 401,
                    'message' => 'This order was not payed with Atrato',
                )
            );
        }
        return json_encode(
            array(
                'code' => 400,
                'message' => 'Headers where not found or are wrong in the request',
            )
        );
    }

    /**
     * Post change order status to complete or cancelled
     * @param string $id
     * @return string
     */
    public function changeOrderStatus(string $id): string
    {
        // Get the header from the request
        $authenticated = $this->AuthValidation();
        // Validate that they exist
        if ($authenticated) {
            // Get the order
            $order = $this->orderRepository->get($id);
            $payment_method = $order->getPayment()->getMethod();
            if ($payment_method === "atrato_paylater") {
                //Get the body;
                $body = $this->request->getBodyParams();
                $new_status = $body['status'];
                //Set and Update the status
                $order->setStatus($new_status);
                // If a note is sent, notify the user about the status of his order
                if (isset($body['note'])) {
                    $customer_note = $body['note'];
                    $history = $this->historyFactory->create()->setParentId($id)
                        ->setComment($customer_note)->setIsCustomerNotified(false)
                        ->setEntityName("order")->setStatus($new_status);
                    $this->historyRepository->save($history);
                }
                $this->orderRepository->save($order);
                return json_encode(
                    array(
                        'code' => 200,
                        'message' => "Updated order:{$id} to status:{$new_status}",
                    )
                );
            }
            return json_encode(
                array(
                    'code' => 401,
                    'message' => 'This order was not payed with Atrato',
                )
            );
        }
        return json_encode(
            array(
                'code' => 400,
                'message' => 'Headers where not found or are wrong in the request',
            )
        );
    }

    /**
     * @return bool
     */
    private function AuthValidation(): bool
    {
        $basic_auth = $this->request->getHeader('Authorization');
        $auth_token = str_replace("Basic ", "", $basic_auth);
        $auth_token = base64_decode($auth_token);
        $auth_token = explode(":", $auth_token);
        $user = $auth_token[0];
        $password = $auth_token[1];
        if ($user === $this->atratoConfig->getUser() && $password === $this->atratoConfig->getKey()) {
            return true;
        }
        return false;
    }
}
