<?php

namespace Atrato\PayLater\Model\Config;

use Atrato\PayLater\Helper\Data;
use Magento\Checkout\Model\ConfigProviderInterface;
use Magento\Customer\Model\Session;
use Magento\Customer\Model\SessionFactory;
use Magento\Framework\App\Config\ScopeConfigInterface;
use Magento\Framework\Exception\NoSuchEntityException;
use Magento\Store\Model\ScopeInterface;
use Magento\Store\Model\StoreManagerInterface;


class Settings implements ConfigProviderInterface
{
    // Configuration params
    private $ECOMMERCE_PARTNER_KEY;
    private $USER;
    private $KEY;
    private $MINIMUM_AMOUNT;
    private $MAXIMUM_AMOUNT;
    private $URL_BASE_ATRATO;
    private $URL_API_ATRATO;

    const CODE = 'atrato_paylater';
    const CHECKOUT_CONFIG_CODE = 'atrato_paylater';
    const URL_LOGO_ATRATO = 'https://atratotech.com/static/media/logoAtrato.baac30ab.png';

    private $helperData;

    /**
     * @var ScopeConfigInterface
     */
    protected $scopeConfig;
    /**
     * @var StoreManagerInterface
     */
    protected $storeManager;
    /**
     * @var SessionFactory
     */
    private $sessionFactory;


    /**
     * Settings constructor.
     * @param ScopeConfigInterface $scopeConfig
     * @param StoreManagerInterface $storeManager
     * @param SessionFactory $sessionFactory
     */
    public function __construct(
        ScopeConfigInterface $scopeConfig,
        StoreManagerInterface $storeManager,
        SessionFactory $sessionFactory,
        Data $helperData

    ) {
        $this->scopeConfig = $scopeConfig;
        $this->storeManager = $storeManager;
        $this->sessionFactory = $sessionFactory;
        $this->helperData = $helperData;
        // Config params
        $this->setupStoreConfig();
    }

    private function setupStoreConfig()
    {
        try {
            $storeCode = $this->storeManager->getStore()->getCode();
        } catch (NoSuchEntityException $e) {
            return false;
        }
        $this->ECOMMERCE_PARTNER_KEY = $this->helperData->getGeneralConfig('partner_key', $storeCode);
        $this->USER = $this->helperData->getGeneralConfig('user', $storeCode);
        $this->KEY = $this->helperData->getGeneralConfig('key', $storeCode);
        $this->MINIMUM_AMOUNT = (float) $this->helperData->getGeneralConfig('minimum_amount', $storeCode);
        $this->MAXIMUM_AMOUNT = (float) $this->helperData->getGeneralConfig('maximum_amount', $storeCode);
        $this->URL_BASE_ATRATO = ((bool) $this->helperData->getGeneralConfig('sandbox')) ? 'https://sandbox.atratopago.com' : 'https://atratopago.com';
        $this->URL_API_ATRATO = ((bool) $this->helperData->getGeneralConfig('sandbox')) ? 'https://api-sandbox.atratopago.com' : 'https://api.atratopago.com';
    }

    /**
     * @param $paymentMethodCode
     * @return bool
     */
    public function getEnabledForStores()
    {;
        try {
            $storeCode = $this->storeManager->getStore()->getCode();
        } catch (NoSuchEntityException $e) {
            return false;
        }
        return (bool)$this->helperData->getGeneralConfig('active', $storeCode);
    }

    /**
     * @return string
     * @throws NoSuchEntityException
     */
    public function getCurrentStoreCode()
    {
        return $this->storeManager->getStore()->getCode();
    }

    /**
     * @param $xmlPath
     * @return mixed
     */
    protected function getValue($xmlPath)
    {
        return $this->scopeConfig->getValue($xmlPath, ScopeInterface::SCOPE_STORE);
    }

    /**
     * @return string
     */
    public function getLogo()
    {
        return self::URL_LOGO_ATRATO;
    }

    /**
     * @return string
     */
    public function getPartnerKey()
    {
        return $this->ECOMMERCE_PARTNER_KEY;
    }

    /**
     * @return string
     */
    public function getUser()
    {
        return $this->USER;
    }

    /**
     * @return string
     */
    public function getKey()
    {
        return $this->KEY;
    }

    /**
     * @return string
     */
    public function getId()
    {
        return base64_encode($this->getUser() . ':' . $this->getKey());
    }

    /**
     * @return string
     */
    public function getUrlBaseAtrato()
    {
        return $this->URL_BASE_ATRATO;
    }

    /**
     * @return string
     */
    public function getUrlApiAtrato()
    {
        return $this->URL_API_ATRATO;
    }

    public function getMinimumAmount()
    {
        return $this->MINIMUM_AMOUNT;
    }

    public function getMaximumAmount()
    {
        return $this->MAXIMUM_AMOUNT;
    }

    /**
     * @inheritDoc
     * @return array
     */
    public function getConfig()
    {
        /**
         * @var Session $customer
         */
        $customer = $this->sessionFactory->create();
        return [
            'payment' => [
                self::CHECKOUT_CONFIG_CODE => [
                    'enable' => $this->getEnabledForStores(), //&& $customer->getCustomerId(),
                    'logo' => $this->getLogo(),
                    'partnerkey' => $this->getPartnerKey(),
                    'urlbaseatrato' => $this->getUrlBaseAtrato(),
                    'urlapiatrato' => $this->getUrlApiAtrato(),
                    'id' => $this->getId(),
                ]
            ]
        ];
    }
}
