<?php
/**
 * @copyright: Copyright © 2017 Firebear Studio. All rights reserved.
 * @author   : Firebear Studio <fbeardev@gmail.com>
 */

namespace Firebear\ImportExport\Console\Command;

use Exception;
use Firebear\ImportExport\Api\JobRepositoryInterface;
use Firebear\ImportExport\Logger\Logger;
use Firebear\ImportExport\Model\Job;
use Firebear\ImportExport\Model\Job\Handler\HandlerInterface;
use Firebear\ImportExport\Model\Job\Handler\HandlerPoolInterface;
use Firebear\ImportExport\Model\Job\Processor;
use Firebear\ImportExport\Model\JobFactory;
use Firebear\ImportExport\Model\Source\Type\AbstractType;
use Firebear\ImportExport\Model\Source\Type\SearchSourceTypeInterface;
use Firebear\ImportExport\Model\Source\Type\SourceTypeInterface;
use Magento\Backend\App\Area\FrontNameResolver;
use Magento\Framework\App\State;
use Magento\Framework\Console\Cli;
use Magento\Framework\Exception\FileSystemException;
use Magento\Framework\Exception\LocalizedException;
use Symfony\Component\Console\Input\InputArgument;
use Symfony\Component\Console\Input\InputInterface;
use Symfony\Component\Console\Output\OutputInterface;
use Firebear\ImportExport\Helper\Data as Helper;

/**
 * Command prints list of available currencies
 */
class ImportJobRunCommand extends ImportJobAbstractCommand
{
    protected $importSource;

    /**
     * @var HandlerPoolInterface
     */
    protected $handlerPool;

    /**
     * Constructor
     *
     * @param JobFactory $factory
     * @param JobRepositoryInterface $repository
     * @param Logger $logger
     * @param Processor $importProcessor
     * @param Helper $helper
     * @param State $state
     * @param HandlerPoolInterface $handlerPool
     */
    public function __construct(
        JobFactory $factory,
        JobRepositoryInterface $repository,
        Logger $logger,
        Processor $importProcessor,
        Helper $helper,
        State $state,
        HandlerPoolInterface $handlerPool
    ) {
        $this->handlerPool = $handlerPool;

        parent::__construct(
            $factory,
            $repository,
            $logger,
            $importProcessor,
            $helper,
            $state
        );
    }

    /**
     * {@inheritdoc}
     */
    protected function configure()
    {
        $this->setName('import:job:run')
            ->setDescription('Generate Firebear Import Jobs')
            ->setDefinition(
                [
                    new InputArgument(
                        self::JOB_ARGUMENT_NAME,
                        InputArgument::OPTIONAL | InputArgument::IS_ARRAY,
                        'Space-separated list of import job ids or omit to generate all jobs.'
                    )
                ]
            );

        parent::configure();
    }

    /**
     * {@inheritdoc}
     */
    protected function execute(InputInterface $input, OutputInterface $output)
    {
        $time = explode(" ", microtime());
        $startTime = $time[0] + $time[1];

        $isAreaCode = 0;
        try {
            if ($this->state->getAreaCode()) {
                $isAreaCode = 1;
            }
        } catch (Exception $e) {
            $isAreaCode = 0;
        }
        if (!$isAreaCode) {
            $this->state->setAreaCode(FrontNameResolver::AREA_CODE);
        }
        $requestedIds = $input->getArgument(self::JOB_ARGUMENT_NAME);
        $requestedIds = array_filter(array_map('trim', $requestedIds), 'strlen');
        $jobCollection = $this->factory->create()->getCollection();
        $jobCollection->addFieldToFilter('is_active', 1);
        if ($requestedIds) {
            $jobCollection->addFieldToFilter('entity_id', ['in' => $requestedIds]);
        }
        if ($jobCollection->getSize()) {
            /** @var Job $job */
            foreach ($jobCollection as $job) {
                $noProblems = 0;
                $result = false;
                $id = (int)$job->getEntityId();
                $file = $this->helper->beforeRun($id);
                $history = $this->helper->createHistory($id, $file, 'console');
                $this->processor->debugMode = $this->debugMode = $this->helper->getDebugMode();
                $this->processor->inConsole = 1;
                $this->processor->setLogger($this->helper->getLogger());
                $sourceData = $job->getSourceData();
                $importSourceType = $this->getImportSource($sourceData);
                if (isset($sourceData[SearchSourceTypeInterface::SCAN_DIR])
                    && $sourceData[SearchSourceTypeInterface::SCAN_DIR] == 1
                    && $importSourceType->isSearchable()
                ) {
                    try {
                        $this->addLogComment(__('Scan Directory for files'), $output, 'info');
                        $filePath = $importSourceType->getImportFilePath() ?: $sourceData['file_path'] ?? '';
                        $fileType = $sourceData['type_file'];

                        if (!$importSourceType->isExists($filePath)) {
                            throw new LocalizedException(__('Path not exists'));
                        }
                        if (!$importSourceType->isAllowablePath($filePath)) {
                            throw new LocalizedException(__(
                                'Path not allowable. Only %1',
                                implode(', ', $importSourceType->allowedDirsForScan())
                            ));
                        }

                        $files = $importSourceType->search("$filePath/*.$fileType");
                        if (!empty($files)) {
                            $files = $importSourceType
                                ->filterSearchedFiles($files);
                            foreach ($files as $importFile) {
                                $importFile = $importSourceType
                                    ->getFilePath($importFile);
                                $file = $this->helper->beforeRun($id);
                                $history = $this->helper->createHistory($id, $file, 'console');
                                $this->addLogComment(__('Import File %1', $importFile), $output, 'info');
                                try {
                                    $this->processor->setImportFile($importFile);
                                    [$noProblems, $result] = $this->processImport($id, $file, $job, $history);
                                    if (!$this->processor->getIsValidated()) {
                                        $noProblems = 1;
                                    }
                                } catch (Exception $exception) {
                                    $this->addLogComment(
                                        $exception->getMessage(),
                                        $output,
                                        'error'
                                    );
                                    $noProblems = 1;
                                }
                                $f = pathinfo($importFile);
                                if ($noProblems) {
                                    if (strpos($this->processor->getErrorMessage(), 'Connection Lost') !== false) {
                                        $this->addLogComment(
                                            __('Please re-run the job due to connection lost'),
                                            $output,
                                            'error'
                                        );
                                        return false;
                                    }
                                    $path = $importSourceType
                                            ->getConfigFilePath() . $importSourceType
                                            ->getDirName(SearchSourceTypeInterface::ERR_DIR);
                                } else {
                                    $path = $importSourceType
                                            ->getConfigFilePath() . $importSourceType->getDirName();
                                }
                                $this->addLogComment(
                                    __('Moving file to %1 after import', $path . $f['basename']),
                                    $output,
                                    'info'
                                );
                                $importSourceType->move($importFile, $path . $f['basename']);
                                $this->postImportProcess($job, $file, $result);
                            }
                        }
                    } catch (LocalizedException | FileSystemException $localizedException) {
                        $this->addLogComment(
                            __('Error occured detailed error in logger file'),
                            $output,
                            'error'
                        );
                        $this->logger->error($localizedException->getMessage());
                    }
                } else {
                    try {
                        [$noProblems, $result] = $this->processImport($id, $file, $job, $history);
                    } catch (LocalizedException $localizedException) {
                        $this->addLogComment(
                            $localizedException->getMessage(),
                            $output,
                            'error'
                        );
                    }
                }
                if (!$noProblems && $this->processor->reindex) {
                    $this->processor->processReindex($file, $id);
                }
                $this->postImportProcess($job, $file, $result);
            }
        } else {
            $this->addLogComment('No jobs found', $output, 'error');
        }

        $time = explode(" ", microtime());
        $endTime = $time[0] + $time[1];
        $totalTime = $endTime - $startTime;
        $totalTime = round($totalTime, 5);
        $this->addLogComment("------------" . $totalTime, $output, 'info');
        return Cli::RETURN_SUCCESS;
    }

    /**
     * @param array $sourceData
     * @return AbstractType|SourceTypeInterface|SearchSourceTypeInterface
     * @throws LocalizedException
     */
    protected function getImportSource(array $sourceData)
    {
        if ($this->importSource === null) {
            $this->importSource = $this->processor->getImportModel()
                ->setData($sourceData)
                ->getSource();
        }
        return $this->importSource;
    }

    /**
     * @param int $id
     * @param string $file
     * @param $job
     * @param $history
     * @return array
     * @throws LocalizedException
     */
    protected function processImport(int $id, string $file, $job, $history)
    {
        $noProblems = 0;
        $this->processor->processScope($id, $file);
        $counter = $this->helper->countData($file, $job->getId());
        $result = $error = 0;
        for ($i = 0; $i < $counter; $i++) {
            [$count, $result] = $this->helper->processImport($file, $job->getId(), $i, $error, 0);
            $error += $count;
            if (!$result) {
                $noProblems = 1;
                break;
            }
        }
        $this->processor->showErrors();
        $this->processor->getImportModel()->getErrorAggregator()->clear();
        $this->processor->getImportModel()->setNullEntityAdapter();
        $this->helper->saveFinishHistory($history);
        return [$noProblems, $result];
    }

    /**
     * @param Job $job
     * @param string $file
     * @param bool $result
     * @throws LocalizedException
     */
    protected function postImportProcess(Job $job, string $file, bool $result)
    {
        /** @var HandlerInterface $handler */
        foreach ($this->handlerPool->getHandlersInstances() as $handler) {
            $handler->execute($job, $file, (int)$result);
        }
    }
}
