<?php
/**
 * Copyright © MageWorx. All rights reserved.
 * See LICENSE.txt for license details.
 */

namespace MageWorx\SeoRedirects\Model\Redirect\CustomRedirect\Export;

use MageWorx\SeoRedirects\Api\Data\CustomRedirectInterface;

class CsvDataProvider
{
    /**
     * @var \MageWorx\SeoRedirects\Model\ResourceModel\Redirect\CustomRedirect\CollectionFactory
     */
    protected $redirectCollectionFactory;

    /**
     * @var \MageWorx\SeoRedirects\Model\Redirect\Source\RedirectTypeEntityCode
     */
    protected $redirectTypeEntityCodeOptions;

    /**
     * CsvDataProvider constructor.
     *
     * @param \MageWorx\SeoRedirects\Model\Redirect\Source\RedirectTypeEntityCode $redirectTypeEntityCodeOptions
     * @param \MageWorx\SeoRedirects\Model\ResourceModel\Redirect\CustomRedirect\CollectionFactory $redirectCollectionFactory
     */
    public function __construct(
        \MageWorx\SeoRedirects\Model\Redirect\Source\RedirectTypeEntityCode $redirectTypeEntityCodeOptions,
        \MageWorx\SeoRedirects\Model\ResourceModel\Redirect\CustomRedirect\CollectionFactory $redirectCollectionFactory
    ) {
        $this->redirectTypeEntityCodeOptions = $redirectTypeEntityCodeOptions;
        $this->redirectCollectionFactory     = $redirectCollectionFactory;
    }

    /**
     * @param array|null $redirectIds
     * @return string
     */
    public function getContent(array $redirectIds = null)
    {
        /** start csv content and set template */
        $headers = new \Magento\Framework\DataObject(
            [
                CustomRedirectInterface::REQUEST_ENTITY_TYPE       => __('Request Entity Type'),
                CustomRedirectInterface::REQUEST_ENTITY_IDENTIFIER => __('Request Entity Identifier'),
                CustomRedirectInterface::TARGET_ENTITY_TYPE        => __('Target Entity Type'),
                CustomRedirectInterface::TARGET_ENTITY_IDENTIFIER  => __('Target Entity Identifier'),
                CustomRedirectInterface::REDIRECT_CODE             => __('Redirect Type'),
                CustomRedirectInterface::STORE_ID                  => __('Store ID'),
                CustomRedirectInterface::START_AT                  => __('Active From'),
                CustomRedirectInterface::FINISH_AT                 => __('Active To'),
                CustomRedirectInterface::DATE_CREATED              => __('Created'),
                CustomRedirectInterface::DATE_MODIFIED             => __('Updated'),
                CustomRedirectInterface::IS_AUTOGENERATED          => __('Is Autogenerated'),
                CustomRedirectInterface::IS_IMPORTED               => __('Is Imported'),
                CustomRedirectInterface::STATUS                    => __('Enable'),
            ]
        );

        $template =
            '"{{' . CustomRedirectInterface::REQUEST_ENTITY_TYPE . '}}",' .
            '"{{' . CustomRedirectInterface::REQUEST_ENTITY_IDENTIFIER . '}}",' .
            '"{{' . CustomRedirectInterface::TARGET_ENTITY_TYPE . '}}",' .
            '"{{' . CustomRedirectInterface::TARGET_ENTITY_IDENTIFIER . '}}",' .
            '"{{' . CustomRedirectInterface::REDIRECT_CODE . '}}",' .
            '"{{' . CustomRedirectInterface::STORE_ID . '}}",' .
            '"{{' . CustomRedirectInterface::START_AT . '}}",' .
            '"{{' . CustomRedirectInterface::FINISH_AT . '}}",' .
            '"{{' . CustomRedirectInterface::DATE_CREATED . '}}",' .
            '"{{' . CustomRedirectInterface::DATE_MODIFIED . '}}",' .
            '"{{' . CustomRedirectInterface::IS_AUTOGENERATED . '}}",' .
            '"{{' . CustomRedirectInterface::IS_IMPORTED . '}}",' .
            '"{{' . CustomRedirectInterface::STATUS . '}}"';;

        $content = $headers->toString($template);

        $content .= "\n";

        $collection = $this->getCustomRedirectCollection($redirectIds);

        /** @var \MageWorx\SeoRedirects\Model\Redirect\CustomRedirect $redirect */
        while ($redirect = $collection->fetchItem()) {
            $content .= $this->toExportString($redirect, $template) . "\n";
        }

        return $content;
    }

    /**
     * Convert redirect into string with predefined format
     *
     * Will use $format as an template and substitute {{key}} for attributes
     *
     * @param \MageWorx\SeoRedirects\Model\Redirect\CustomRedirect $redirect
     * @param string $format
     * @return string
     */
    protected function toExportString($redirect, $format)
    {
        preg_match_all('/\{\{([a-z0-9_]+)\}\}/is', $format, $matches);
        foreach ($matches[1] as $var) {
            $format = str_replace('{{' . $var . '}}', $this->getExportValue($redirect, $var), $format);
        }

        return $format;
    }

    /**
     * @param \MageWorx\SeoRedirects\Model\Redirect\CustomRedirect $redirect
     * @param string $var
     * @return mixed
     */
    protected function getExportValue($redirect, $var)
    {
        switch ($var) {
            case CustomRedirectInterface::REQUEST_ENTITY_TYPE:
            case CustomRedirectInterface::TARGET_ENTITY_TYPE:
                $value = $this->redirectTypeEntityCodeOptions->toArray()[$redirect->getData($var)];
                break;

            default:
                $value = $redirect->getData($var);
        }

        return $value;
    }

    /**
     * @param array/null $redirectIds
     * @return \MageWorx\SeoRedirects\Model\ResourceModel\Redirect\CustomRedirect\Collection
     */
    protected function getCustomRedirectCollection(array $redirectIds = null)
    {
        /** @var \MageWorx\SeoRedirects\Model\ResourceModel\Redirect\CustomRedirect\Collection $redirectCollection */
        $redirectCollection = $this->redirectCollectionFactory->create();

        if ($redirectIds) {
            $redirectCollection->addFieldToFilter(CustomRedirectInterface::REDIRECT_ID, $redirectIds);
        }

        return $redirectCollection;
    }
}
