<?php
/**
 * Copyright © MageWorx. All rights reserved.
 * See LICENSE.txt for license details.
 */

namespace MageWorx\SeoReports\Model\Trigger;

use MageWorx\SeoReports\Model\GeneratorFactory;
use Magento\UrlRewrite\Service\V1\Data\UrlRewrite;
use Magento\CmsUrlRewrite\Model\CmsPageUrlRewriteGenerator;


class PageReportTrigger
{
    /**
     * @var \MageWorx\SeoReports\Model\Config\Page
     */
    protected $pageConfig;

    /**
     * @var GeneratorFactory
     */
    protected $generatorFactory;

    /**
     * @var \MageWorx\SeoReports\Model\ResourceModel\PageReport
     */
    protected $reportResource;

    /**
     * @var \Magento\UrlRewrite\Model\UrlFinderInterface
     */
    protected $urlFinder;

    /**
     * @var \Magento\Store\Model\StoreManagerInterface
     */
    protected $storeManager;

    /**
     * PageReportTrigger constructor.
     *
     * @param \Magento\Store\Model\StoreManagerInterface $storeManager
     * @param \MageWorx\SeoReports\Model\Config\Page $pageConfig
     * @param GeneratorFactory $generatorFactory
     * @param \MageWorx\SeoReports\Model\ResourceModel\PageReport $reportResource
     * @param \Magento\UrlRewrite\Model\UrlFinderInterface $urlFinder
     */
    public function __construct(
        \Magento\Store\Model\StoreManagerInterface $storeManager,
        \MageWorx\SeoReports\Model\Config\Page $pageConfig,
        \MageWorx\SeoReports\Model\GeneratorFactory $generatorFactory,
        \MageWorx\SeoReports\Model\ResourceModel\PageReport $reportResource,
        \Magento\UrlRewrite\Model\UrlFinderInterface $urlFinder
    ) {
        $this->storeManager     = $storeManager;
        $this->pageConfig       = $pageConfig;
        $this->generatorFactory = $generatorFactory;
        $this->reportResource   = $reportResource;
        $this->urlFinder        = $urlFinder;
    }

    /**
     * @param \Magento\Cms\Model\Page $page
     */
    public function generateReportForNewPage($page)
    {
        $origPageStoreId = $page->getStoreId();
        $pageStoreIds    = $page->getStoreId() == ['0'] ? array_keys(
            $this->storeManager->getStores()
        ) : $page->getStoreId();
        $requestPaths    = $this->getPageUrlsByStores($page, $pageStoreIds);

        foreach ($pageStoreIds as $storeId) {
            $requestPath = !empty($requestPaths[$storeId]) ? $requestPaths[$storeId] : null;
            $page->setData('request_path', $requestPath);
            $page->setStoreId($storeId);
            $this->getGenerator()->generate($page);
        }

        $page->setStoreId($origPageStoreId);
    }

    /**
     * @param \Magento\Cms\Model\Page $page
     */
    public function generateReportForPage($page)
    {
        $origStoreId = $page->getStoreId();

        $existReportStoreIds = $this->reportResource->getStoresForReference($page->getId());
        $storeIds            = $page->getStores() == ['0'] ? array_keys(
            $this->storeManager->getStores()
        ) : $page->getStores();

        $missedStoreIdsInReports = array_diff($storeIds, $existReportStoreIds);
        $deletedStoreIds         = array_diff($existReportStoreIds, $storeIds);

        if ($missedStoreIdsInReports) {
            $page->setStoreId($missedStoreIdsInReports);
            $this->generateReportForNewPage($page);
            $page->setStoreId($origStoreId);
        }

        $data = $this->getModifiedData($page);

        foreach ($storeIds as $storeId) {
            $data['id']       = $page->getId();
            $data['store_id'] = $storeId;
            $object           = new \Magento\Framework\DataObject($data);

            $this->getGenerator()->generate($object);
        }

        if ($deletedStoreIds) {
            $this->getGenerator()->regenerateByRemoval($page, $storeIds);
        }
    }

    /**
     * @param \Magento\Cms\Model\Page $page
     */
    public function regenerateReportForRemovalPage($page)
    {
        $this->getGenerator()->regenerateByRemoval($page);
    }

    /**
     * @return \MageWorx\SeoReports\Model\Generator\PageReportGenerator
     */
    protected function getGenerator()
    {
        /** @var \MageWorx\SeoReports\Model\Generator\PageReportGenerator $generator */
        $generator = $this->generatorFactory->create('cms_page');

        return $generator;
    }

    /**
     * @param \Magento\Cms\Model\Page $page
     * @param int[] $stores
     * @return array
     */
    protected function getPageUrlsByStores($page, $stores)
    {
        $filterData = [
            UrlRewrite::ENTITY_ID        => $page->getId(),
            UrlRewrite::ENTITY_TYPE      => CmsPageUrlRewriteGenerator::ENTITY_TYPE,
            UrlRewrite::IS_AUTOGENERATED => true,
        ];

        $urlRewrites = $this->urlFinder->findAllByData($filterData);

        $result = [];

        foreach ($urlRewrites as $urlRewrite) {

            $storeId = $urlRewrite->getStoreId();
            if (in_array((int)$storeId, $stores)) {
                $result[$storeId] = $urlRewrite->getRequestPath();
            }
        }

        return $result;
    }

    /**
     * @param \Magento\Cms\Model\Page $page
     * @return array
     */
    protected function getModifiedData($page)
    {
        $changedFields = [];

        foreach ($this->getFieldList() as $field) {
            if ($page->dataHasChangedFor($field)) {
                $changedFields[$field] = $page->getData($field);
            }
        }

        return $changedFields;
    }

    /**
     * @return array
     */
    protected function getFieldList()
    {
        return $this->pageConfig->getFieldList();
    }
}