<?php
/**
 * Copyright © Magento, Inc. All rights reserved.
 * See COPYING.txt for license details.
 */
namespace Mageants\AskForPrice\Block\CustomerData;

use Magento\Catalog\Model\Product\Image\NotLoadInfoImageException;
use Magento\Customer\CustomerData\SectionSourceInterface;

/**
 * Wishlist section
 */
class Wishlist extends \Magento\Wishlist\CustomerData\Wishlist
{
    /**
     * @var string
     */
    const SIDEBAR_ITEMS_NUMBER = 3;

    /**
     * @var \Magento\Wishlist\Helper\Data
     */
    protected $wishlistHelper;

    /**
     * @var \Magento\Catalog\Helper\ImageFactory
     */
    protected $imageHelperFactory;

    /**
     * @var \Magento\Framework\App\ViewInterface
     */
    protected $view;

    /**
     * @var \Magento\Wishlist\Block\Customer\Sidebar
     */
    protected $block;
    /**
     * @var \Mageants\AskForPrice\Helper\Data
     */
    protected $helperData;

    /**
     * @param \Magento\Wishlist\Helper\Data $wishlistHelper
     * @param \Magento\Wishlist\Block\Customer\Sidebar $block
     * @param \Magento\Catalog\Helper\ImageFactory $imageHelperFactory
     * @param \Magento\Framework\App\ViewInterface $view
     * @param \Mageants\AskForPrice\Helper\Data $helperData
     */
    public function __construct(
        \Magento\Wishlist\Helper\Data $wishlistHelper,
        \Magento\Wishlist\Block\Customer\Sidebar $block,
        \Magento\Catalog\Helper\ImageFactory $imageHelperFactory,
        \Magento\Framework\App\ViewInterface $view,
        \Mageants\AskForPrice\Helper\Data $helperData
    ) {
        $this->wishlistHelper = $wishlistHelper;
        $this->imageHelperFactory = $imageHelperFactory;
        $this->block = $block;
        $this->view = $view;
        $this->helperData = $helperData;
    }

    /**
     * {@inheritdoc}
     */
    public function getSectionData()
    {
        $counter = $this->getCounter();
        return [
            'counter' => $counter,
            'items' => $counter ? $this->getItems() : [],
        ];
    }

    /**
     * @return string
     */
    protected function getCounter()
    {
        return $this->createCounter($this->wishlistHelper->getItemCount());
    }

    /**
     * Create button label based on wishlist item quantity
     *
     * @param int $count
     * @return \Magento\Framework\Phrase|null
     */
    protected function createCounter($count)
    {
        if ($count > 1) {
            return __('%1 items', $count);
        } elseif ($count == 1) {
            return __('1 item');
        }
        return null;
    }

    /**
     * Get wishlist items
     *
     * @return array
     */
    protected function getItems()
    {
        $this->view->loadLayout();

        $collection = $this->wishlistHelper->getWishlistItemCollection();
        $collection->clear()->setPageSize(self::SIDEBAR_ITEMS_NUMBER)
            ->setInStockFilter(true)->setOrder('added_at');

        $items = [];
        foreach ($collection as $wishlistItem) {
            $items[] = $this->getItemData($wishlistItem);
        }
        return $items;
    }

    /**
     * Retrieve wishlist item data
     *
     * @param \Magento\Wishlist\Model\Item $wishlistItem
     * @return array
     */
    protected function getItemData(\Magento\Wishlist\Model\Item $wishlistItem)
    {
        $product = $wishlistItem->getProduct();
        return [
            'image' => $this->getImageData($product),
            'product_url' => $this->wishlistHelper->getProductUrl($wishlistItem),
            'product_name' => $product->getName(),
            'product_id' => $product->getId(),
            'product_sku' => $product->getSku(),
            'product_priority' => $this->getPriority($product),
            'product_label' => $this->getProductLabel($product),
            'product_askforprice_text' => $product->getAskforpriceText(),
            'product_customer_groups' => $this->getCustomerGroups($product),
            'priority_category_ids' => $this->getCategoryIds($product),
            'product_cat_ids' => $this->getProductCatIds($product),
            'not_apply_ids' => $this->getNotApplyIds($product),
            'product_price' => $this->block->getProductPriceHtml(
                $product,
                'wishlist_configured_price',
                \Magento\Framework\Pricing\Render::ZONE_ITEM_LIST,
                ['item' => $wishlistItem]
            ),
            'product_is_saleable_and_visible' => $product->isSaleable() && $product->isVisibleInSiteVisibility(),
            'product_has_required_options' => $product->getTypeInstance()->hasRequiredOptions($product),
            'add_to_cart_params' => $this->wishlistHelper->getAddToCartParams($wishlistItem, true),
            'delete_item_params' => $this->wishlistHelper->getRemoveParams($wishlistItem, true),
        ];
    }
    protected function getPriority($product){
        $priority = $this->helperData->getPriority($product);
        return $priority;
    }
    protected function getProductLabel($product){
        $label=$this->helperData->getLabel($product,$this->getPriority($product));
        return $label;
    }
    protected function getCustomerGroups($product){
        $priority = $this->getPriority($product);
        $customerGroup = $this->helperData->getCustomerGroups($product,$priority);
        $customerGroups = implode(',', $customerGroup);
        return $customerGroups;
    }
    protected function getCategoryIds($product){
        $priority = $this->getPriority($product);
        $category_id = $this->helperData->getCategoryIds($priority);
        $category_ids = implode(',', $category_id);
        return $category_ids;
    }
    protected function getProductCatIds($product){
        $product_cat_id = $product->getCategoryIds();
        $product_cat_ids = implode(',', $product_cat_id);
        return $product_cat_ids;
    }
    protected function getNotApplyIds($product){
        $priority = $this->getPriority($product);
        $notApplyIds=$this->helperData->getNotApplyIds($priority);
        return $notApplyIds;
    }
    /**
     * Retrieve product image data
     *
     * @param \Magento\Catalog\Model\Product $product
     * @return array
     * @SuppressWarnings(PHPMD.NPathComplexity)
     */
    protected function getImageData($product)
    {
        /*Set variant product if it is configurable product.
        It will show variant product image in sidebar instead of configurable product image.*/
        $simpleOption = $product->getCustomOption('simple_product');
        if ($simpleOption !== null) {
            $optionProduct = $simpleOption->getProduct();
            $product = $optionProduct;
        }

        /** @var \Magento\Catalog\Helper\Image $helper */
        $helper = $this->imageHelperFactory->create()
            ->init($product, 'wishlist_sidebar_block');

        $template = $helper->getFrame()
            ? 'Magento_Catalog/product/image'
            : 'Magento_Catalog/product/image_with_borders';

        try {
            $imagesize = $helper->getResizedImageInfo();
        } catch (NotLoadInfoImageException $exception) {
            $imagesize = [$helper->getWidth(), $helper->getHeight()];
        }

        $width = $helper->getFrame()
            ? $helper->getWidth()
            : $imagesize[0];

        $height = $helper->getFrame()
            ? $helper->getHeight()
            : $imagesize[1];

        return [
            'template' => $template,
            'src' => $helper->getUrl(),
            'width' => $width,
            'height' => $height,
            'alt' => $helper->getLabel(),
        ];
    }
}
