<?php

/**
 * Olegnax MegaMenu
 *
 * This file is part of Olegnax/Core.
 *
 * NOTICE OF LICENSE
 *
 * This source file is subject to the Olegnax.com license that is
 * available through the world-wide-web at this URL:
 * https://www.olegnax.com/license
 *
 * DISCLAIMER
 *
 * Do not edit or add to this file if you wish to upgrade this extension to newer
 * version in the future.
 *
 * @category    Olegnax
 * @package     Olegnax_MegaMenu
 * @copyright   Copyright (c) 2019 Olegnax (http://www.olegnax.com/)
 * @license     https://www.olegnax.com/license
 */

namespace Olegnax\MegaMenu\Block\Html;

use Magento\Catalog\Model\CategoryFactory;
use Magento\Cms\Model\Template\FilterProvider;
use Magento\Framework\App\ObjectManager;
use Magento\Framework\Data\Tree\Node;
use Magento\Framework\Exception\NoSuchEntityException;
use Magento\Framework\UrlInterface;
use Magento\Store\Model\ScopeInterface;
use Magento\Store\Model\StoreManagerInterface;
use Magento\Theme\Block\Html\Topmenu;

class Megamenu extends Topmenu
{
    /**
     * @var string
     */
    const BASE_IMAGE_PATH = "catalog/category/";
    /**
     * @var string
     */
    private $_mediaUrl;

    /**
     * Get relevant path to template
     *
     * @return string
     */
    public function getTemplate()
    {
        if ($this->isEnabled()) {
            return 'Olegnax_MegaMenu::megamenu.phtml';
        }
        return $this->_template;
    }

    /**
     * @return bool
     */
    protected function isEnabled()
    {
        return (bool)$this->getValueOption('enable_megamenu');
    }

    /**
     * @param $path
     * @param string $default
     * @return mixed|string
     */
    public function getValueOption($path, $default = '')
    {
        if ($this->hasData($path)) {
            return $this->getData($path);
        }
        $value = $this->getConfig($path);
        if (is_null($value)) {
            $value = $default;
        }

        return $value;
    }

    /**
     * @param string $path
     * @param string $storeCode
     * @return mixed
     */
    public function getConfig($path, $storeCode = null)
    {
        return $this->getSystemValue('ox_megamenu_settings/general/' . $path, $storeCode);
    }

    /**
     * @param string $path
     * @param string $storeCode
     * @return mixed
     */
    public function getSystemValue($path, $storeCode = null)
    {
        return $this->_scopeConfig->getValue($path, ScopeInterface::SCOPE_STORE, $storeCode);
    }

    /**
     * Get cache key informative items
     *
     * @return array
     * @since 100.1.0
     */
    public function getCacheKeyInfo()
    {
        $keyInfo = parent::getCacheKeyInfo();
        $keyInfo[] = $this->getUrl('*/*/*', ['_current' => true, '_query' => '']);
        return $keyInfo;
    }

    /**
     * @param Node $item
     * @param string $outermostClassCode
     * @param string $childrenWrapClass
     * @param int $limit
     * @param string $colBrakes
     * @param bool $is_megamenu
     * @param int $childLevel
     * @return string
     */
    protected function _getHtml_lvl0(
        Node $item,
        $outermostClassCode,
        $childrenWrapClass,
        $limit,
        $colBrakes,
        $is_megamenu = false,
        $childLevel = 0
    ) {
        $hasChildren = $item->hasChildren();
        $category = $this->getCategory($item);
        $is_megamenu = (bool)$category->getData('ox_nav_type');
        $navContent = [];
        $item->setData('is_megamenu', $is_megamenu);
        $this->set_custom_class($item, $category);

        $style = [];
        $content = $category->getData('ox_bg_image');
        if (!empty($content)) {
            $style['background-image'] = 'url(' . $this->getModuleMediaUrl($content) . ')';
        }
        $style = $this->prepareStyle($style);
        $style = $style ? ' style="' . $style . '"' : '';

        $html = '';
        if ($is_megamenu) {
            $navContent = array_filter([
                'top' => $this->getBlockTemplateProcessor($category->getData('ox_nav_top')),
                'left' => $this->getBlockTemplateProcessor($category->getData('ox_nav_left')),
                'right' => $this->getBlockTemplateProcessor($category->getData('ox_nav_right')),
                'bottom' => $this->getBlockTemplateProcessor($category->getData('ox_nav_btm')),
            ]);
            $columns = $category->getData('ox_columns');
            if ($hasChildren || !empty($navContent)) {

                $html .= '<div class="ox-megamenu__dropdown"' . $this->prepareAttributes([
                        'data-ox-mm-w' => $category->getData('ox_menu_width'),
                        'data-ox-mm-cw' => $category->getData('ox_nav_column_width'),
                        'data-ox-mm-col' => $category->getData('ox_columns'),
                    ]) . $style . '>';
                $layout = $category->getData('ox_layout');
                switch ($layout) {
                    case 2:
                        $html .= '<div class="row">';
                        if (isset($navContent['left'])) {
                            $html .= '<div class="ox-megamenu-block ox-megamenu-block-left ox-menu-col ox-menu-col-' . $category->getData('ox_nav_left_width') . '">' . $navContent['left'] . '</div>';
                        }
                        $html .= '<div class="ox-menu-col">';
                        if (isset($navContent['top'])) {
                            $html .= '<div class="ox-megamenu-block ox-megamenu-block-top">' . $navContent['top'] . '</div>';
                        }
                        if ($hasChildren && !$category->getData('ox_nav_subcategories')) {
                            $html .= '<div class="ox-megamenu-block ox-megamenu__categories">';
                            if ($columns > 0) {
                                $columns = 'row ox-megamenu-list--columns-' . $columns;
                            }
                            $html .= '<ul class="ox-megamenu-list ' . $columns . '">';
                            $html .= $this->_addSubMenu($item, $childLevel, $childrenWrapClass, $limit, $is_megamenu);
                            $html .= '</ul>';
                            $html .= '</div>';
                        }
                        if (isset($navContent['bottom'])) {
                            $html .= '<div class="ox-megamenu-block ox-megamenu-block-bottom">' . $navContent['bottom'] . '</div>';
                        }
                        $html .= '</div>'; //close column
                        if (isset($navContent['right'])) {
                            $html .= '<div class="ox-megamenu-block ox-megamenu-block-right ox-menu-col ox-menu-col-' . $category->getData('ox_nav_right_width') . '">' . $navContent['right'] . '</div>';
                        }
                        $html .= '</div>'; //close row
                        break;
                    case 3:
                        $html .= '<div class="row">';
                        if (isset($navContent['left'])) {
                            $html .= '<div class="ox-megamenu-block ox-megamenu-block-left ox-menu-col ox-menu-col-' . $category->getData('ox_nav_left_width') . '">' . $navContent['left'] . '</div>';
                        }
                        $html .= '<div class="ox-menu-col">';
                        if (isset($navContent['top'])) {
                            $html .= '<div class="ox-megamenu-block ox-megamenu-block-top">' . $navContent['top'] . '</div>';
                        }
                        if ($hasChildren && !$category->getData('ox_nav_subcategories')) {
                            $html .= '<div class="ox-megamenu-block ox-megamenu__categories">';
                            $columns = $category->getData('ox_columns');
                            if ($columns > 0) {
                                $columns = 'row ox-megamenu-list--columns-' . $columns;
                            }
                            $html .= '<ul class="ox-megamenu-list ' . $columns . '">';
                            $html .= $this->_addSubMenu($item, $childLevel, $childrenWrapClass, $limit, $is_megamenu);
                            $html .= '</ul>';
                            $html .= '</div>';
                        }
                        $html .= '</div>'; // close column
                        if (isset($navContent['right'])) {
                            $html .= '<div class="ox-megamenu-block ox-megamenu-block-right ox-menu-col ox-menu-col-' . $category->getData('ox_nav_right_width') . '">' . $navContent['right'] . '</div>';
                        }
                        $html .= '</div>'; // close row
                        if (isset($navContent['bottom'])) {
                            $html .= '<div class="ox-megamenu-block ox-megamenu-block-bottom">' . $navContent['bottom'] . '</div>';
                        }
                        break;
                    case 4:
                        if (isset($navContent['top'])) {
                            $html .= '<div class="ox-megamenu-block ox-megamenu-block-top">' . $navContent['top'] . '</div>';
                        }
                        $html .= '<div class="row">';
                        if (isset($navContent['left'])) {
                            $html .= '<div class="ox-megamenu-block ox-megamenu-block-left ox-menu-col ox-menu-col-' . $category->getData('ox_nav_left_width') . '">' . $navContent['left'] . '</div>';
                        }
                        $html .= '<div class="ox-menu-col">';
                        if ($hasChildren && !$category->getData('ox_nav_subcategories')) {
                            $html .= '<div class="ox-megamenu-block ox-megamenu__categories">';
                            $columns = $category->getData('ox_columns');
                            if ($columns > 0) {
                                $columns = 'row ox-megamenu-list--columns-' . $columns;
                            }
                            $html .= '<ul class="ox-megamenu-list ' . $columns . '">';
                            $html .= $this->_addSubMenu($item, $childLevel, $childrenWrapClass, $limit, $is_megamenu);
                            $html .= '</ul>';
                            $html .= '</div>';
                        }
                        if (isset($navContent['bottom'])) {
                            $html .= '<div class="ox-megamenu-block ox-megamenu-block-bottom">' . $navContent['bottom'] . '</div>';
                        }
                        $html .= '</div>'; // close column
                        if (isset($navContent['right'])) {
                            $html .= '<div class="ox-megamenu-block ox-megamenu-block-right ox-menu-col ox-menu-col-' . $category->getData('ox_nav_right_width') . '">' . $navContent['right'] . '</div>';
                        }
                        $html .= '</div>'; // close row

                        break;
                    default:
                        if (isset($navContent['top'])) {
                            $html .= '<div class="ox-megamenu-block ox-megamenu-block-top">' . $navContent['top'] . '</div>';
                        }
                        if (($hasChildren && !$category->getData('ox_nav_subcategories')) || isset($navContent['left']) || isset($navContent['right'])) {
                            $html .= '<div class="row">';
                            if (isset($navContent['left'])) {
                                $html .= '<div class="ox-megamenu-block ox-megamenu-block-left ox-menu-col ox-menu-col-' . $category->getData('ox_nav_left_width') . '">' . $navContent['left'] . '</div>';
                            }
                            if ($hasChildren && !$category->getData('ox_nav_subcategories')) {
                                $html .= '<div class="ox-megamenu-block ox-megamenu__categories ox-menu-col">';
                                $columns = $category->getData('ox_columns');
                                if ($columns > 0) {
                                    $columns = 'row ox-megamenu-list--columns-' . $columns;
                                }
                                $html .= '<ul class="ox-megamenu-list ' . $columns . '">';
                                $html .= $this->_addSubMenu($item, $childLevel, $childrenWrapClass, $limit,
                                    $is_megamenu);
                                $html .= '</ul>';
                                $html .= '</div>';
                            }
                            if (isset($navContent['right'])) {
                                $html .= '<div class="ox-megamenu-block ox-megamenu-block-right ox-menu-col ox-menu-col-' . $category->getData('ox_nav_right_width') . '">' . $navContent['right'] . '</div>';
                            }
                            $html .= '</div>'; //close row
                        }
                        if (isset($navContent['bottom'])) {
                            $html .= '<div class="ox-megamenu-block ox-megamenu-block-bottom">' . $navContent['bottom'] . '</div>';
                        }
                }

                $html .= '</div>';
            }
        } else {
            if ($hasChildren) {

                $menuWidth = $category->getData('ox_menu_width');
                $menuWidth = $menuWidth ? 'data-ox-mm-w="' . $menuWidth . '"' : '';
                $html .= '<div class="ox-megamenu__dropdown" ' . $menuWidth . $style . '><ul class="ox-megamenu-list">' . $this->_addSubMenu($item,
                        $childLevel, $childrenWrapClass, $limit) . '</ul></div>';
            }
        }

        $html_a = $this->getCatCLContent($category) . $this->getItemName($item) . $this->getCatLabel($category);
        if ($hasChildren || !empty($navContent)) {
            $html_a .= $this->add_parent_arrow($this->getConfig('show_menu_parent_arrow'));
        }

        $html = '<li ' . $this->_getRenderedMenuItemAttributes($item) . '>' . $this->wrapItemLink($html_a, $item,
                $category, $outermostClassCode) . $html . '</li>';

        return $html;
    }

    /**
     * @param Node $item
     * @return mixed
     */
    protected function getCategory($item)
    {
        return $this->_getCategory(str_replace('category-node-', '', $item->getId()));
    }

    /**
     * @param int $id
     * @return CategoryFactory
     */
    protected function _getCategory($id)
    {
        return $this->_loadObject(CategoryFactory::class)->create()->load($id);
    }

    /**
     * @param string $object
     * @return mixed
     */
    protected function _loadObject($object)
    {
        return $this->_getObjectManager()->get($object);
    }

    /**
     * @return ObjectManager
     */
    protected function _getObjectManager()
    {
        return ObjectManager::getInstance();
    }

    /**
     * @param Node $item
     * @param $category
     */
    protected function set_custom_class(Node $item, $category)
    {
        $custom_class = trim($category->getData('ox_nav_custom_class'));
        if (!empty($custom_class)) {
            $item->setData('class', trim($item->getClass() . ' ' . $custom_class));
        }
    }

    /**
     * @param string $path
     * @return string
     * @throws NoSuchEntityException
     */
    public function getModuleMediaUrl($path = '')
    {
        return $this->getBaseMediaUrl() . $path;
    }

    /**
     * @return string
     * @throws NoSuchEntityException
     */
    protected function getBaseMediaUrl()
    {
        if (!$this->_mediaUrl) {
            /** @var StoreManagerInterface $storeManager */
            $storeManager = $this->_loadObject(StoreManagerInterface::class);
            $this->_mediaUrl = $storeManager->getStore()->getBaseUrl(UrlInterface::URL_TYPE_MEDIA) .
                static::BASE_IMAGE_PATH;
        }

        return $this->_mediaUrl;
    }

    /**
     * @param array $style
     * @param string $separatorValue
     * @param string $separatorAttribute
     * @return string
     */
    public function prepareStyle(array $style, $separatorValue = ': ', $separatorAttribute = ';')
    {
        $style = array_filter($style);
        if (empty($style)) {
            return '';
        }
        foreach ($style as $key => &$value) {
            $value = $key . $separatorValue . $value;
        }
        $style = implode($separatorAttribute, $style);

        return $style;
    }

    /**
     * @param string $content
     * @return string
     */
    protected function getBlockTemplateProcessor($content = '')
    {
        return $this->_loadObject(FilterProvider::class)->getBlockFilter()->filter(trim($content));
    }

    /**
     * @param array $attributes
     * @return string
     */
    public function prepareAttributes(array $attributes)
    {
        $attributes = array_filter($attributes);
        if (empty($attributes)) {
            return '';
        }
        $html = '';
        foreach ($attributes as $attributeName => $attributeValue) {
            $html .= ' ' . $attributeName . '="' . str_replace('"', '\"', $attributeValue) . '"';
        }
        return $html;
    }

    /**
     * Add sub menu HTML code for current menu item
     *
     * @param Node $child
     * @param string $childLevel
     * @param string $childrenWrapClass
     * @param int $limit
     * @param bool $is_megamenu
     *
     * @return string HTML code
     */
    protected function _addSubMenu($child, $childLevel, $childrenWrapClass, $limit, $is_megamenu = false)
    {
        if (!$this->isEnabled()) {
            return parent::_addSubMenu($child, $childLevel, $childrenWrapClass, $limit);
        }
        if (!$child->hasChildren()) {
            return '';
        }

        $colStops = [];
        if ($childLevel == 0 && $limit) {
            $colStops = $this->_columnBrake($child->getChildren(), $limit);
        }
        $customClass = '';
        if (($is_megamenu && $childLevel > 1) || (!$is_megamenu && $childLevel > 0)) {
            $customClass = 'ox-submenu';
        }
        $html = $this->_getHtml($child, $childrenWrapClass, $limit, $colStops, $is_megamenu);
        if ($childLevel > 0) {
            $html = '<ul class="' . $customClass . ' level' . $childLevel . ' ' . $childrenWrapClass . '">' . $html . '</ul>';
        }

        return $html;
    }

    /**
     * @param Node $menuTree
     * @param string $childrenWrapClass
     * @param int $limit
     * @param array $colBrakes
     * @param bool $is_megamenu
     * @return string
     */
    protected function _getHtml(
        Node $menuTree,
        $childrenWrapClass,
        $limit,
        array $colBrakes = [],
        $is_megamenu = false
    ) {
        if (!$this->isEnabled()) {
            return parent::_getHtml($menuTree, $childrenWrapClass, $limit, $colBrakes);
        }
        $html = '';

        $children = $menuTree->getChildren();
        $parentLevel = $menuTree->getLevel();
        $childLevel = $parentLevel === null ? 0 : $parentLevel + 1;

        $counter = 0;
        $itemPosition = 0;
        $childrenCount = $children->count();

        $parentPositionClass = $menuTree->getPositionClass();

        /** @var Node $child */
        foreach ($children as $child) {
            if ($childLevel === 0 && $child->getData('is_parent_active') === false) {
                continue;
            }
            $itemPosition++;
            $counter++;
            $child->setLevel($childLevel);
            $child->setIsFirst($counter == 1);
            $child->setIsLast($counter == $childrenCount);


            $outermostClassCode = '';
            $outermostClass = $menuTree->getOutermostClass();

            if ($childLevel == 0 && $outermostClass) {
                $outermostClassCode = ' class="' . $outermostClass . '" ';
                $currentClass = $child->getClass();

                if (empty($currentClass)) {
                    $child->setClass($outermostClass);
                } else {
                    $child->setClass($currentClass . ' ' . $outermostClass);
                }
            }

            $fucntion = '_getHtml_lvlx';
            switch ($childLevel) {
                case 0:
                    $fucntion = '_getHtml_lvl0';
                    break;
                case 1:
                    $fucntion = '_getHtml_lvl1';
                    break;
            }
            $html .= $this->$fucntion($child, $outermostClassCode, $childrenWrapClass, $limit, $colBrakes, $is_megamenu,
                $childLevel);

        }

        return $html;
    }

    /**
     * @param $category
     * @return string
     */
    protected function getCatCLContent($category)
    {
        $content = $category->getData('ox_nav_custom_link_content');
        if ($content) {
            return '<span class="ox-menu-item__custom-element">' . $content . '</span>';
        }
        return '';
    }

    /**
     * @param Node $item
     * @return string
     */
    protected function getItemName(Node $item)
    {
        return '<span class="name">' . $this->escapeHtml($item->getName()) . '</span>';
    }

    /**
     * @param $category
     * @return string
     */
    protected function getCatLabel($category)
    {
        $content = $category->getData('ox_category_label');
        if ($content) {
            return '<span class="ox-megamenu-label" style="' . $this->prepareStyle([
                    'color' => $category->getData('ox_label_text_color'),
                    'background-color' => $category->getData('ox_label_color')
                ]) . '">' . $content . '</span>';
        }
        return '';
    }

    /**
     * @param bool $showSubCat
     * @return string
     */
    protected function add_parent_arrow($showSubCat)
    {
        if ($showSubCat) {
            return '<i class="ox-menu-arrow"></i>';
        } else {
            return '<i class="ox-menu-arrow hide-on-desktop"></i>';
        }
    }

    /**
     * @param string $html
     * @param Node $item
     * @param $category
     * @param string $outermostClassCode
     * @return string
     */
    protected function wrapItemLink($html, Node $item, $category, $outermostClassCode)
    {
        $custom_link_target = $category->getData('ox_nav_custom_link_target') ? 'target="_blank" ' : '';
        $style = $this->prepareStyle([
            'color' => $category->getData('ox_title_text_color'),
            'background-color' => $category->getData('ox_title_bg_color'),
        ]);
        $style = $style ? ' style="' . $style . '"' : '';
        return '<a ' . $custom_link_target . 'href="' . $this->getItemCustomLink($item,
                $category) . '" ' . $outermostClassCode . $style . '>' . $html . '</a>';
    }

    /**
     * @param Node $item
     * @param $category
     * @return mixed
     */
    protected function getItemCustomLink(Node $item, $category)
    {
        $custom_url = $category->getData('ox_nav_custom_link');
        if ($custom_url) {
            return $custom_url;
        }
        return $item->getUrl();
    }

    /**
     * @param Node $item
     * @param string $outermostClassCode
     * @param string $childrenWrapClass
     * @param int $limit
     * @param string $colBrakes
     * @param bool $is_megamenu
     * @param int $childLevel
     * @return string
     */
    protected function _getHtml_lvl1(
        Node $item,
        $outermostClassCode,
        $childrenWrapClass,
        $limit,
        $colBrakes,
        $is_megamenu = false,
        $childLevel = 1
    ) {
        $category = $this->getCategory($item);
        $this->set_custom_class($item, $category);

        $html = $this->getCatCLContent($category);

        $content = $category->getData('ox_cat_image');
        if (!empty($content)) {
            $html .= '<span class="ox-menu__category-image"><img src="' . $this->getModuleMediaUrl($content) . '"></span>';
        }

        $html .= $this->getItemName($item) . $this->getCatLabel($category);

        if ($item->hasChildren()) {
            $html .= $this->add_parent_arrow($this->getConfig('show_sub_parent_arrow'));
        }

        $html = '<li ' . $this->_getRenderedMenuItemAttributes($item) . '>' . $this->wrapItemLink($html, $item,
                $category, $outermostClassCode) . $this->_addSubMenu($item, $childLevel, $childrenWrapClass, $limit,
                $is_megamenu) . '</li>';

        return $html;
    }

    /**
     * @param Node $item
     * @param string $outermostClassCode
     * @param string $childrenWrapClass
     * @param int $limit
     * @param string $colBrakes
     * @param bool $is_megamenu
     * @param int $childLevel
     * @return string
     */
    protected function _getHtml_lvlx(
        Node $item,
        $outermostClassCode,
        $childrenWrapClass,
        $limit,
        $colBrakes,
        $is_megamenu = false,
        $childLevel = 2
    ) {
        $category = $this->getCategory($item);
        $this->set_custom_class($item, $category);

        $html = $this->getCatCLContent($category) . $this->getItemName($item);

        if ($item->hasChildren()) {
            $html .= $this->add_parent_arrow($this->getConfig('show_sub_parent_arrow'));
        }
        $html .= $this->getCatLabel($category);

        $html = '<li ' . $this->_getRenderedMenuItemAttributes($item) . '>' . $this->wrapItemLink($html, $item,
                $category, $outermostClassCode) . $this->_addSubMenu($item, $childLevel, $childrenWrapClass, $limit,
                $is_megamenu) . '</li>';

        return $html;
    }

    /**
     * Returns array of menu item's attributes
     *
     * @param Node $item
     * @return array
     */
    protected function _getMenuItemAttributes(Node $item)
    {
        if (!$this->isEnabled()) {
            return parent::_getMenuItemAttributes($item);
        }
        $menuItemClasses = $this->_getMenuItemClasses($item);
        $menuItemAttributes = ['class' => implode(' ', $menuItemClasses)];
        if ($this->getCategory($item)->getData('ox_data_tm_align_horizontal') && 0 == $item->getLevel()) {
            $menuItemAttributes['data-ox-mm-a-h'] = $this->getCategory($item)->getData('ox_data_tm_align_horizontal');
        }
        return $menuItemAttributes;
    }

    /**
     * Returns array of menu item's classes
     *
     * @param Node $item
     * @return array
     */
    protected function _getMenuItemClasses(Node $item)
    {
        $classes = parent::_getMenuItemClasses($item);
        if (!$this->isEnabled()) {
            return $classes;
        }
        if (0 == $item->getLevel()) {
            $classes[] = 'ox-dropdown--' . ($item->getData('is_megamenu') ? 'megamenu' : 'simple');
        }

        return $classes;
    }

}
