<?php

namespace Kueski;

use Kueski\Data\Billing;
use Kueski\Data\Item;
use Kueski\Data\Metadata;
use Kueski\Data\Amount;
use Kueski\Data\Shipping;

class Kueski
{

    /* URL CONSTANTS */

    const URL_PRODUCTION = "https://api.kueskipay.com/";
    const URL_SANDBOX = "https://testing.kueskipay.com/";

    /**
     * @var Amount
     */
    private $amount = null;

    /**
     * @var Item[]
     */
    private $items = array();

    /**
     * @var Metadata
     */
    private $metadata = null;

    /**
     * @var string
     */
    private $callbackSuccess = "";

    /**
     * @var string
     */
    private $callbackReject = "";

    /**
     * @var string
     */
    private $callbackCanceled = "";

    /**
     * @var string
     */
    private $callbackFailed = "";

    /**
     * @var string
     */
    private $orderId;

    /**
     * @var string
     */
    private $description;

    /**
     * @var bool
     */
    private $sandbox;

    /**
     * @var string
     */
    private $apiKey;

    /**
     * @var Shipping
     */
    private $shippingAddress;

    /**
     * @var Billing
     */
    private $billingAddress;

    /**
     * @var string
     */
    private $kpName;

    /**
     * @var string
     */
    private $kpVersion;

    /**
     * @var string
     */
    private $kpTrigger;

    /**
     * @var string
     */
    private $kpSource;

    /**
     * Kueski constructor.
     * @param string $orderId
     * @param string $description
     * @param string $apiKey
     * @param string $callbackSuccess
     * @param string $callbackReject
     * @param string $callbackCanceled
     * @param string $callbackFailed
     * @param $kpName
     * @param $kpVersion
     * @param $kpTrigger
     * @param $kpSource
     * @param bool $sandbox
     */
    public function __construct(
        $orderId,
        $description,
        $apiKey,
        $callbackSuccess,
        $callbackReject,
        $callbackCanceled,
        $callbackFailed,
        $kpName,
        $kpVersion,
        $kpTrigger,
        $kpSource,
        $sandbox = false
    )
    {
        $this->apiKey = $apiKey;
        $this->callbackSuccess = $callbackSuccess;
        $this->callbackReject = $callbackReject;
        $this->callbackCanceled = $callbackCanceled;
        $this->callbackFailed = $callbackFailed;
        $this->orderId = $orderId;
        $this->description = $description;
        $this->sandbox = $sandbox;
        $this->kpName = $kpName;
        $this->kpVersion = $kpVersion;
        $this->kpTrigger = $kpTrigger;
        $this->kpSource = $kpSource;
    }

    /**
     * @param string $serviceCode
     * @param string $description
     * @return Kueski
     */
    public function addNewAdditionalService($serviceCode = "", $description = "") {
        $this->getMetadata()->addNewAdditionalServiceData($serviceCode, $description);
        return $this;
    }

    /**
     * @param string $name
     * @param string $description
     * @param string $sku
     * @param float $quantity
     * @param float $price
     * @param string $currency
     * @param float $tax
     * @return Kueski
     */
    public function addNewItem($name, $description, $sku, $quantity, $price, $currency, $tax = 0.00) {
        $item = new Item();
        $item->setName($name);
        $item->setDescription($description);
        $item->setSku($sku);
        $item->setQuantity($quantity);
        $item->setPrice($price);
        $item->setCurrency($currency);
        $item->setTax($tax);

        $this->items[]  = $item;
        return $this;
    }

    /**
     * @param string $carrierCode
     * @param string $flightNumber
     * @param string $departureStation
     * @param string $departureTime
     * @param string $arrivalStation
     * @param string $arrivalTime
     * @return Kueski
     */
    public function addNewJourney(
        $carrierCode = "",
        $flightNumber = "",
        $departureStation = "",
        $departureTime = "",
        $arrivalStation = "",
        $arrivalTime = ""
    ) {
        $this->getMetadata()->addNewJourneyData($carrierCode, $flightNumber, $departureStation, $departureTime, $arrivalStation, $arrivalTime);
        return $this;
    }

    /**
     * @param string $firstName
     * @param string $middleName
     * @param string $lastName
     * @param string $birthdate
     * @param string $nationality
     * @param string $gender
     * @param string $email
     * @param string $mobile
     * @param string $phone
     * @return Kueski
     */
    public function addNewPassenger(
        $firstName = "",
        $middleName = "",
        $lastName = "",
        $birthdate = "",
        $nationality = "",
        $gender = "",
        $email = "",
        $mobile = "",
        $phone = ""
    ) {
        $this->getMetadata()->addNewPassengerData($firstName, $middleName, $lastName, $birthdate, $nationality, $gender, $email, $mobile, $phone);
        return $this;
    }

    /**
     * @param float $subtotal
     * @param float $shipping
     * @param float $handlingFee
     * @param float $tax
     * @param string $currency
     * @return Kueski
     */
    public function setAmountDetails($subtotal, $shipping = 0.00, $handlingFee = 0.00, $tax = 0.00, $currency = "MXN")
    {
        $amount = $this->getAmount();
        $amount->addDetails($subtotal, $shipping, $handlingFee, $tax);
        $total = $subtotal + $shipping + $handlingFee + $tax;
        $amount->setCurrency($currency);
        $amount->setTotal($total);
        $this->amount = $amount;
        return $this;
    }

    /**
     * @param $name
     * @param $email
     * @param $mobile
     * @param $phone
     * @param $postalCode
     * @return Kueski
     */
    public function setContact($name, $email, $mobile, $phone, $postalCode) {
        $this->getMetadata()->setContact($name, $email, $mobile, $phone, $postalCode);
        return $this;
    }

    /**
     * @param $name
     * @param $lastname
     * @param $address
     * @param $interior
     * @param $neighborhood
     * @param $city
     * @param $state
     * @param $zipcode
     * @param $country
     * @param $phoneNumber
     * @param $email
     * @param $type
     * @return $this
     */
    public function setShippingAddress(
        $name,
        $lastname,
        $address,
        $interior,
        $neighborhood,
        $city,
        $state,
        $zipcode,
        $country,
        $phoneNumber,
        $email,
        $type
    ) {
        $this->getShippingAddress()->setName($name, $lastname);
        $this->getShippingAddress()->setAddress($address, $interior, $neighborhood, $city, $state, $zipcode, $country);
        $this->getShippingAddress()->setAdditionalInformation($phoneNumber, $email, $type);
        return $this;
    }

    /**
     * @param $name
     * @param $rfc
     * @param $address
     * @param $interior
     * @param $neighborhood
     * @param $city
     * @param $state
     * @param $zipcode
     * @param $country
     * @param $phoneNumber
     * @param $email
     * @return $this
     */
    public function setBillingAddress(
        $name,
        $rfc,
        $address,
        $interior,
        $neighborhood,
        $city,
        $state,
        $zipcode,
        $country,
        $phoneNumber,
        $email
    ) {
        $this->getBillingAddress()->setName($name, $rfc);
        $this->getBillingAddress()->setAddress($address, $interior, $neighborhood, $city, $state, $zipcode, $country);
        $this->getBillingAddress()->setAdditionalInformation($phoneNumber, $email);
        return $this;
    }

    /**
     * @return Amount
     */
    public function getAmount()
    {
        if ( $this->amount == null ) {
            $this->amount = new Amount();
        }
        return $this->amount;
    }

    /**
     * @return array
     */
    public function getItems()
    {
        if ( $this->items == null ) {
            $this->items = array();
        }
        return $this->items;
    }

    /**
     * @return Metadata
     */
    public function getMetadata()
    {
        if ( $this->metadata == null ) {
            $this->metadata = new Metadata();
        }
        return $this->metadata;
    }

    /**
     * @return Shipping
     */
    public function getShippingAddress()
    {
        if ( $this->shippingAddress == null ) {
            $this->shippingAddress = new Shipping();
        }
        return $this->shippingAddress;
    }

    /**
     * @return Billing
     */
    public function getBillingAddress()
    {
        if ( $this->billingAddress == null ) {
            $this->billingAddress = new Billing();
        }
        return $this->billingAddress;
    }

    /**
     * @return array
     */
    public function getShippingAddressArray()
    {
        $retVal = [];

        $retVal["name"] = $this->getShippingAddress()->getName();
        $retVal["address"] = $this->getShippingAddress()->getAddress();
        $retVal["phone_number"] = $this->getShippingAddress()->getPhoneNumber();
        $retVal["email"] = $this->getShippingAddress()->getEmail();
        $retVal["type"] = $this->getShippingAddress()->getType();
        return $retVal;
    }

    /**
     * @return array
     */
    public function getAmountArray()
    {
        $retVal = [];

        $retVal["total"] = $this->getAmount()->getTotal();
        $retVal["currency"] = $this->getAmount()->getCurrency();
        $retVal["details"]["subtotal"] = $this->getAmount()->getDetails()->getSubtotal();
        $retVal["details"]["shipping"] = $this->getAmount()->getDetails()->getShipping();
        $retVal["details"]["handling_fee"] = $this->getAmount()->getDetails()->getHandlingFee();
        $retVal["details"]["tax"] = $this->getAmount()->getDetails()->getTax();
        return $retVal;
    }

    /**
     * @return array
     */
    public function getBillingAddressArray()
    {
        $retVal = [];

        $retVal["business"] = $this->getBillingAddress()->getBusiness();
        $retVal["address"] = $this->getBillingAddress()->getAddress();
        $retVal["phone_number"] = $this->getBillingAddress()->getPhoneNumber();
        $retVal["email"] = $this->getBillingAddress()->getEmail();
        return $retVal;
    }

    /**
     * @return array
     */
    public function makeJSONArray() {
        $json = array();
        $json["description"] = $this->getDescription();
        $json["order_id"] = $this->getOrderId();
        $json["callbacks"]["on_success"] = $this->getCallbackSuccess();
        $json["callbacks"]["on_reject"] = $this->getCallbackReject();
        $json["callbacks"]["on_canceled"] = $this->getCallbackCanceled();
        $json["callbacks"]["on_failed"] = $this->getCallbackFailed();
        $json["amount"] = $this->getAmountArray();
        $json["items"] = $this->getItemsArray();
        $json["shipping"] = $this->getShippingAddressArray();
        $json["billing"] = $this->getBillingAddressArray();

        return $json;
    }

    /**
     * @return array
     */
    public function getPaymentArray() {
        $json = array();
        $json["callbacks"]["on_success"] = $this->getCallbackSuccess();
        $json["callbacks"]["on_reject"] = $this->getCallbackReject();
        $json["callbacks"]["on_canceled"] = $this->getCallbackCanceled();
        $json["callbacks"]["on_failed"] = $this->getCallbackFailed();

        return $json;
    }

    /**
     * @return false|string
     */
    public function getJSON() {
        return json_encode($this->makeJSONArray());
    }

    /**
     * @return array
     */
    protected function getItemsArray() {
        $items = [];
        $count = 0;
        /** @var Item $item */
        foreach ($this->getItems() as $item) {
            $items[$count]["name"] = $item->getName();
            $items[$count]["description"] = $item->getDescription();
            $items[$count]["quantity"] = $item->getQuantity();
            $items[$count]["price"] = $item->getPrice();
            $items[$count]["currency"] = $item->getCurrency();
            $items[$count]["tax"] = $item->getTax();
            $items[$count]["sku"] = $item->getSku();
            $count++;
        }
        return $items;
    }

    /**
     * @return string
     */
    public function getDescription()
    {
        return $this->description;
    }

    /**
     * @return string
     */
    public function getCallbackSuccess()
    {
        return $this->callbackSuccess;
    }

    /**
     * @return string
     */
    public function getCallbackReject()
    {
        return $this->callbackReject;
    }

    /**
     * @return string
     */
    public function getCallbackCanceled()
    {
        return $this->callbackCanceled;
    }

    /**
     * @return string
     */
    public function getCallbackFailed()
    {
        return $this->callbackFailed;
    }

    /**
     * @return string
     */
    public function getOrderId()
    {
        return $this->orderId;
    }

    /**
     * @return string
     */
    public function getApiKey()
    {
        return $this->apiKey;
    }

    /**
     * @return bool
     */
    public function isSandbox()
    {
        return $this->sandbox;
    }

    /**
     * @return string
     */
    public function getUrl() {
        if ($this->isSandbox()) {
            return self::URL_SANDBOX;
        } else {
            return self::URL_PRODUCTION;
        }
    }

    public function isApiKeyValid() {
        $ch = curl_init();

        $url = $this->getUrl() . "v1/validate-keys?api_key=" . $this->getApiKey();

        curl_setopt($ch, CURLOPT_URL, $url);
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_CONNECTTIMEOUT, 0);
        curl_setopt($ch, CURLOPT_TIMEOUT, 0);
        $result = curl_exec($ch);
        curl_close($ch);

        if (!$result) {
            return false;
        }

        $result = json_decode($result);
        if ($result->status == "fail") {
            return false;
        }

        return true;
    }

    /**
     * @return false|string
     * @throws \Exception
     */
    public function sendOrder() {

        if (!$this->isApiKeyValid()) {
            throw new \Exception("Unauthorized Api Key");
        }

        $ch = curl_init();

        $post = $this->getJSON();
        $header = array();

        $header[] = "Authorization: Bearer " . $this->getApiKey();
        $header[] = 'Content-length: ' . strlen($post);
        $header[] = 'Content-type: application/json';
        $header[] = 'kp-name: ' . $this->kpName;
        $header[] = 'kp-version: ' . $this->kpVersion;
        $header[] = 'kp-source: ' . $this->kpSource;
        $header[] = 'kp-trigger: ' . $this->kpTrigger;
        curl_setopt($ch, CURLOPT_URL, $this->getUrl() . "v1/payments");
        curl_setopt($ch, CURLOPT_HTTPHEADER, $header);
        curl_setopt($ch, CURLOPT_CUSTOMREQUEST, "POST");
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_POSTFIELDS, $post);
        //curl_setopt($ch, CURLOPT_FOLLOWLOCATION, 1);
        $result = curl_exec($ch);
        curl_close($ch);


        if (!$result) {
            throw new \Exception("Invalid Format");
        }

        $result = json_decode($result);
        if ($result->status == "fail" || $result->status == "error") {
            throw new \Exception($result->message);
        }

        return $result->data->callback_url;
    }
}
