<?php
namespace Kueski\Payment\Controller;

use Exception;
use Magento\Checkout\Model\Cart;
use Magento\Checkout\Model\Session;
use Magento\Framework\App\Action\Action;
use Magento\Framework\App\Action\Context;
use Magento\Framework\App\ActionInterface;
use Magento\Framework\App\ResponseInterface;
use Magento\Framework\Controller\Result\Redirect;
use Magento\Framework\Exception\LocalizedException;
use Magento\Framework\Url\Helper\Data as UrlHelper;
use Magento\Sales\Model\Order;
use Magento\Sales\Model\OrderFactory;

abstract class AbstractKueski extends Action
{

    /**
     * @var OrderFactory
     */
    protected $_orderFactory;
    /**
     * @var UrlHelper
     */
    protected $_urlHelper;
    /**
     * @var \Kueski\Payment\Logger\Logger
     */
    protected $_logger;
    /**
     * @var Cart
     */
    protected $_cart;
    /**
     * @var \Magento\Checkout\Model\Session
     */
    protected $_checkoutSession;

    /**
     * Index constructor.
     * @param UrlHelper $urlHelper
     * @param OrderFactory $orderFactory
     * @param \Kueski\Payment\Logger\Logger $logger
     * @param Cart $cart
     * @param Session $checkoutSession
     * @param Context $context
     */
    public function __construct(
        UrlHelper $urlHelper,
        OrderFactory $orderFactory,
        \Kueski\Payment\Logger\Logger $logger,
        \Magento\Checkout\Model\Cart $cart,
        \Magento\Checkout\Model\Session $checkoutSession,
        Context $context
    ) {
        parent::__construct($context);
        $this->_orderFactory = $orderFactory;
        $this->_urlHelper = $urlHelper;
        $this->_logger = $logger;
        $this->_cart = $cart;
        $this->_checkoutSession = $checkoutSession;
    }

    /**
     * Execute action based on request and return result
     *
     * Note: Request will be added as operation argument in future
     *
     * @return ResponseInterface|Redirect
     * @throws Exception
     */
    public function reOrder()
    {
        $lastOrderId = $this->getRequest()->getParams();
        $lastOrderId = $lastOrderId["id"];

        /** @var Order $_order */
        $_order   = $this->_orderFactory->create()->loadByIncrementId($lastOrderId);
        if ($_order->getId() == null && $_order->getId() < 1 ) {
            $this->_redirect("/");
        }

        if ($_order->canCancel()) {
            $_order->cancel();
            $_order->hasCustomerNoteNotify(true);
            $_order->save();
        }

        $data = array();
        $data[ActionInterface::PARAM_NAME_URL_ENCODED] = $this->_urlHelper->getEncodedUrl();
        return $this->addToCart($_order);
    }

    /**
     * @param Order $order
     * @return string
     */
    protected function getReorderUrl($order)
    {
        return 'sales/order/reorder/order_id/' . $order->getId();
    }

    /**
     * Renders Success
     *
     * @param string $coreRoute
     * @throws Exception
     */
    //@codingStandardsIgnoreLine
    public function success($coreRoute = null)
    {
        $lastOrderId = $this->getRequest()->getParam('id');

        /** @var Order $_order */
        $_order   = $this->_orderFactory->create()->loadByIncrementId($lastOrderId);
        if ($_order->getId() == null && $_order->getId() < 1 ) {
            $this->_redirect("/");
        }

        try {
            $this->loadLayout();
            $this->renderLayout();
        } catch (\Exception $e ) {
            $this->_logger->error($e->getMessage());
        }
    }

    /**
     * @param Order $order
     * @return Redirect
     */
    protected function addToCart($order)
    {
        /** @var Redirect $resultRedirect */
        $resultRedirect = $this->resultRedirectFactory->create();

        $items = $order->getItemsCollection();
        foreach ($items as $item) {
            try {
                $this->_cart->addOrderItem($item);
            } catch (LocalizedException $e) {
                if ($this->_checkoutSession->getUseNotice(true)) {
                    $this->messageManager->addNoticeMessage($e->getMessage());
                } else {
                    $this->messageManager->addErrorMessage($e->getMessage());
                }

                return $resultRedirect->setPath('*/*/history');
            } catch (Exception $e) {
                $this->messageManager->addExceptionMessage(
                    $e,
                    __('We can\'t add this item to your shopping cart right now.')
                );
                return $resultRedirect->setPath('checkout/cart');
            }
        }
        $this->messageManager
            ->addErrorMessage("El proceso de pago con Kueski Pay no fue concluido correctamente.");
        $this->_cart->save();
        return $resultRedirect->setPath('checkout/cart');
    }
}
