<?php
namespace Kueski\Payment\Controller\Index;

use Exception;
use Kueski\Payment\Exceptions\Kueski\ReconciliationException;
use Kueski\Payment\Exceptions\KueskiException;
use Magento\Framework\App\Action\Action;
use Magento\Framework\App\Action\Context;
use Magento\Framework\App\Config\ScopeConfigInterface;
use Magento\Framework\App\ResponseInterface;
use Magento\Framework\Controller\Result\JsonFactory;
use Magento\Framework\Controller\ResultInterface;
use Magento\Sales\Model\Order;
use Magento\Sales\Model\Order\Payment;
use Magento\Sales\Model\Order\PaymentFactory;
use Magento\Sales\Model\OrderFactory;
use Magento\Sales\Model\Service\InvoiceService;
use Magento\Framework\DB\Transaction;
use Magento\Sales\Model\Order\Email\Sender\InvoiceSender;
use Magento\Sales\Model\Order\Email\Sender\OrderSender;
use Magento\Store\Model\ScopeInterface;
use Magento\Sales\Model\Order\Payment\Transaction\BuilderInterface as TransactionBuilder;

class Index extends Action implements \Magento\Framework\App\Action\HttpPostActionInterface
{
    /**
     * @var JsonFactory
     */
    protected $_resultJsonFactory;
    /**
     * @var ScopeConfigInterface
     */
    protected $_scopeConfig;
    /**
     * @var \Kueski\Payment\Logger\Logger
     */
    protected $_logger;
    /**
     * @var OrderFactory
     */
    protected $_orderFactory;
    /**
     * @var PaymentFactory
     */
    protected $_paymentFactory;
    /**
     * @var InvoiceService
     */
    protected $invoiceService;
    /**
     * @var InvoiceSender
     */
    protected $invoiceSender;
    /**
     * @var Transaction
     */
    private $transaction;

    /**
     * @var OrderSender
     */
    protected $_orderSender;
    /**
     * @var \Kueski\Payment\Controller\Index\Payment
     */
    private $payment;
    /**
     * @var \Kueski\Payment\Helper\Data
     */
    private $helper;

    /**
     * Index constructor.
     * @param OrderFactory $orderFactory
     * @param PaymentFactory $paymentFactory
     * @param JsonFactory $resultJsonFactory
     * @param ScopeConfigInterface $scopeConfig
     * @param InvoiceService $invoiceService
     * @param InvoiceSender $invoiceSender
     * @param OrderSender $orderSender
     * @param Transaction $transaction
     * @param \Kueski\Payment\Logger\Logger $logger
     * @param Context $context
     */
    public function __construct(
        \Kueski\Payment\Model\Payment\Kueski          $payment,
        OrderFactory                                  $orderFactory,
        PaymentFactory                                $paymentFactory,
        JsonFactory                                   $resultJsonFactory,
        ScopeConfigInterface                          $scopeConfig,
        InvoiceService                                $invoiceService,
        InvoiceSender                                 $invoiceSender,
        OrderSender                                   $orderSender,
        \Kueski\Payment\Logger\Logger                 $logger,
        \Kueski\Payment\Helper\Data                   $helper,
        TransactionBuilder                            $transaction,
        Context $context
    ) {
        parent::__construct($context);
        $this->payment          = $payment;
        $this->_resultJsonFactory = $resultJsonFactory;
        $this->_scopeConfig = $scopeConfig;
        $this->_logger = $logger;
        $this->_orderFactory = $orderFactory;
        $this->_paymentFactory = $paymentFactory;
        $this->invoiceService = $invoiceService;
        $this->invoiceSender = $invoiceSender;
        $this->transaction = $transaction;
        $this->_orderSender = $orderSender;
        $this->helper = $helper;
    }

    /**
     * Execute action based on request and return result
     *
     * Note: Request will be added as operation argument in future
     *
     * @return ResultInterface|ResponseInterface
     */
    public function execute()
    {
        $jwt = null;

        try {
            $retArray = [];
            // @codingStandardsIgnoreLine
            $phpinput = file_get_contents("php://input");

            $apiSecretKey = $this->_scopeConfig->getValue("payment/kueski_payment/api_secret_key", ScopeInterface::SCOPE_STORE);
            $apiPublicKey = $this->_scopeConfig->getValue("payment/kueski_payment/api_key", ScopeInterface::SCOPE_STORE);

            $this->_logger->info(var_export($phpinput, true));
            $params = json_decode($phpinput);
            $this->_logger->info(var_export($params, true));

            /** @var Order $order */ // @codingStandardsIgnoreLine
            $order = $this->_orderFactory->create()->loadByIncrementId($params->order_id);

            if (!$order->getId()) {
                $retArray = $this->buildKueskiPayResponse("reject", "Order does not exist");
            } else {
                $token = $this->getBearerToken();
                $jwt = $this->generateJwt($apiPublicKey, $apiSecretKey);
                $this->_logger->info(var_export("TOKEN", true));
                $this->_logger->info(var_export($token, true));
                #$this->_logger->info(var_export("DECODE TOKEN", true));
                #$this->_logger->info(var_export(json_decode(base64_decode(str_replace('_', '/', str_replace('-','+',explode('.', $token))))), true));
                $this->_logger->info(var_export("JWT", true));
                $this->_logger->info(var_export($jwt, true));
                $this->_logger->info(var_export("API SECRET JEY", true));
                $this->_logger->info(var_export($apiSecretKey, true));
                $this->_logger->info(var_export("IS VALIDATE KUESKI JWT", true));
                $this->_logger->info(var_export($this->is_jwt_valid($token, $apiSecretKey), true));

                //cuando es texto plano o codeado con jwt
                if ($this->is_jwt_valid($token, $apiSecretKey)) {
                    $retArray = $this->processPaymentReconciliation($order, $params);
                } else {
                    $retArray = $this->processDenied($order, $params, "Auth error: Invalid JWT token.");
                }
            }


            #$resultJson->setHeader('Content-type', 'application/json', true);

            $this->_logger->info(var_export("RESPONSE", true));
            $this->_logger->info(var_export($retArray, true));
            $this->_logger->info(var_export($jwt, true));
        } catch (\Exception $e) {
            $retArray = [
                "status" => 500,
                "error" => __('Not possible to payment reconciliation, critical error, please check logs')
            ];
        } catch (ReconciliationException $e) {
            //Do nothing, yet
        }

        $resultJson = $this->_resultJsonFactory->create();
        $resultJson->setData($retArray)
            ->setHeader("Authorization", "Bearer " . $jwt, true);
        return $resultJson;
    }

    public function generateJwt($apiPublicKey, $apiSecretKey)
    {
        $payloadjwt = [];
        $payloadjwt["public_key"] = $apiPublicKey;
        $payloadjwt["iat"] = strtotime(date('Y-m-d H:i:s'));
        $payloadjwt["exp"] = strtotime("+5 minutes", $payloadjwt["iat"]);
        $payloadjwt["jti"] = hash('sha256', $apiSecretKey . ":" . $payloadjwt["iat"]);

        // Create token header as a JSON string
        $header = json_encode(['typ' => 'JWT', 'alg' => 'HS256']);

        // Create token payload as a JSON string
        $payload = json_encode($payloadjwt);

        // Encode Header to Base64Url String
        $base64UrlHeader = str_replace(['+', '/', '='], ['-', '_', ''], base64_encode($header));

        // Encode Payload to Base64Url String
        $base64UrlPayload = str_replace(['+', '/', '='], ['-', '_', ''], base64_encode($payload));

        // Create Signature Hash
        $signature = hash_hmac('sha256', $base64UrlHeader . "." . $base64UrlPayload, $apiSecretKey, true);

        // Encode Signature to Base64Url String
        $base64UrlSignature = str_replace(['+', '/', '='], ['-', '_', ''], base64_encode($signature));

        // Create JWT
        $jwt = $base64UrlHeader . "." . $base64UrlPayload . "." . $base64UrlSignature;

        $this->_logger->info(var_export("header", true));
        $this->_logger->info(var_export($header, true));
        $this->_logger->info(var_export("payload", true));
        $this->_logger->info(var_export($payload, true));

        return $jwt;
    }

    function is_jwt_valid($jwt, $secret = 'secret')
    {
        // split the jwt
        $tokenParts = explode('.', $jwt);
        $header = base64_decode($tokenParts[0]);
        $payload = base64_decode($tokenParts[1]);
        $signature_provided = $tokenParts[2];

        // check the expiration time - note this will cause an error if there is no 'exp' claim in the jwt
        $expiration = json_decode($payload)->exp;
        $is_token_expired = ($expiration - time()) < 0;

        // build a signature based on the header and payload using the secret
        $base64_url_header = $this->base64url_encode($header);
        $base64_url_payload = $this->base64url_encode($payload);
        $signature = hash_hmac('sha256', $base64_url_header . "." . $base64_url_payload, $secret, true);
        $base64_url_signature = $this->base64url_encode($signature);

        // verify it matches the signature provided in the jwt
        $is_signature_valid = ($base64_url_signature === $signature_provided);

        if ($is_token_expired || !$is_signature_valid) {
            return false;
        } else {
            return true;
        }
    }

    function base64url_encode($str)
    {
        return rtrim(strtr(base64_encode($str), '+/', '-_'), '=');
    }

    /**
     * Get header Authorization
     * */
    function getAuthorizationHeader()
    {
        $headers = null;
        if (isset($_SERVER['Authorization'])) {
            $headers = trim($_SERVER["Authorization"]);
        } else if (isset($_SERVER['HTTP_AUTHORIZATION'])) { //Nginx or fast CGI
            $headers = trim($_SERVER["HTTP_AUTHORIZATION"]);
        } elseif (function_exists('apache_request_headers')) {
            $requestHeaders = apache_request_headers();
            // Server-side fix for bug in old Android versions (a nice side-effect of this fix means we don't care about capitalization for Authorization)
            $requestHeaders = array_combine(array_map('ucwords', array_keys($requestHeaders)), array_values($requestHeaders));
            //print_r($requestHeaders);
            if (isset($requestHeaders['Authorization'])) {
                $headers = trim($requestHeaders['Authorization']);
            }
        }
        return $headers;
    }

    /**
     * get access token from header
     * */
    function getBearerToken()
    {
        $headers = $this->getAuthorizationHeader();
        // HEADER: Get the access token from the header
        if (!empty($headers)) {
            if (preg_match('/Bearer\s(\S+)/', $headers, $matches)) {
                return $matches[1];
            }
        }
        return null;
    }

    /**
     * @param Order $order
     * @param $payload
     * @return array
     */
    protected function processPaymentReconciliation($order, $payload)
    {
        $payloadErrors = $this->validatePayload($order, $payload);
        if (!empty($payloadErrors)) {
            return $this->processDenied($order, $payload, "Kueski payload error: " . $payloadErrors);
        }

        $this->payment->startProcessing($order);

        switch ($payload->status) {
            case "approved": {
                    return $this->processApproved($order, $payload);
                }
            case "denied":
            case "canceled": {
                    return $this->processCanceled($order, $payload);
                }
            default: {
                    return $this->buildKueskiPayResponse("reject", "Incorrect Status");
                }
        }
    }

    /**
     * @param Order $order
     * @param $payload
     * @return string
     */
    protected function validatePayload(Order $order, $payload)
    {
        $errors = [];

        if (abs($order->getGrandTotal() - (float) $payload->amount) >= 1.0) {
            $errors[] = 'Amount is invalid';
        }

        return implode(", ", $errors);
    }

    /**
     * @param Order $order
     * @param $params
     * @return array
     */
    protected function processApproved(Order $order, $params)
    {
        try {
            /** @var Payment $payment */
            $payment = $order->getPayment();
            $message = $params->status_reason;

            $this->payment->processPayment($order);

            if ($order->canInvoice()) {
                $invoice = $this->invoiceService->prepareInvoice($order, []);
                $invoice->addComment(
                    $message,
                    false,
                    false
                );
                $invoice->register();
                $invoice->pay();
                $invoice->setState(\Magento\Sales\Model\Order\Invoice::STATE_PAID);

                $transactionSave = $this->_objectManager->create(
                    \Magento\Framework\DB\Transaction::class
                )->addObject(
                    $invoice
                )->addObject(
                    $invoice->getOrder()
                );

                $transactionSave->save();
                $this->invoiceSender->send($invoice);

                $this->_eventManager->dispatch('kueski_payment_pay_confirmed', [
                    'order' => $order,
                    'payment' => $payment
                ]);
            }



            $this->_logger->info(var_export("TIENE INVOICE?", true));
            $this->_logger->info(var_export($order->canInvoice(), true));
            $this->_logger->info(var_export(!$order->canInvoice(), true));
        } catch (KueskiException $e) {
            return $this->buildKueskiPayResponse("reject", "Error while making invoice. Magento Error: " . $e->getMessage());
        } catch (\Exception $e) {
            return $this->buildKueskiPayResponse("reject", "Error while making invoice. Magento Error: " . $e->getMessage());
        }

        return $this->buildKueskiPayResponse("accept");
    }

    /**
     * @param Order $order
     * @param $params
     * @return array
     */
    protected function processCanceled(Order $order, $params, $comment = null)
    {
        try {
            $reason = !empty($comment) ? $comment : $params->status_reason;
            $canceled = $this->cancelOrderIntent($order, $reason);

            if (!$canceled) {
                return $this->buildKueskiPayResponse("reject", "Order cannot be canceled");
            }

            return $this->buildKueskiPayResponse("ok");
        } catch (Exception $e) {
            return $this->buildKueskiPayResponse("reject", "Error while canceling. Magento Error: " . $e->getMessage());
        }
    }

    /**
     * @param Order $order
     * @param $params
     * @return array
     */
    protected function processDenied(Order $order, $params, $comment = null)
    {
        try {
            $reason = !empty($comment) ? $comment : $params->status_reason;
            $canceled = $this->cancelOrderIntent($order, $reason);

            if (!$canceled) {
                return $this->buildKueskiPayResponse("reject", "Order cannot be canceled");
            }

            return $this->buildKueskiPayResponse("reject", $reason);
        } catch (Exception $e) {
            return $this->buildKueskiPayResponse("reject", "Error while canceling. Magento Error: " . $e->getMessage());
        }
    }

    /**
     * @param Order $order
     * @param string $reason
     * @return boolean
     */
    protected function cancelOrderIntent(Order $order, $reason)
    {
        if (!$order->canCancel()) {
            return false;
        }

        $order->cancel();
        $order->addStatusHistoryComment($reason);
        $order->save();

        return true;
    }

    /**
     * @param string $status
     * @param string $errorMessage
     * @return array
     */
    protected function buildKueskiPayResponse($status, $errorMessage = "")
    {
        /**
         * @ToDo Revisar si el output debe ser en formato plano o si necesita de conversion JSON dentro del controlador
         * @ToDo Refactorizar metodo para fraccionar logica del controlador
         */
        return ["status" => $status, "error" => $errorMessage];
    }
}
