<?php

namespace Kueski\Payment\Model;

use Kueski\Payment\Exceptions\KueskiException;
use Magento\Framework\App\Config\ScopeConfigInterface;
use Magento\Framework\Exception\LocalizedException;
use Magento\Store\Model\ScopeInterface;
use Kueski\Payment\Model\Payment\Integration as KueskiIntegration;

class Config
{
    const XML_PATH_ENABLE           = 'payment/kueski_payment/active';
    const XML_SANDBOX               = 'payment/kueski_payment/sandbox';
    const XML_API_KEY               = 'payment/kueski_payment/api_key';
    const XML_API_SECRET            = 'payment/kueski_payment/api_secret_key';

    // Constraints reserved for future release
    const XML_SUCCESS_LANDING       = 'payment/kueski_payment/success_page';
    const XML_HEAD_DESCRIPTION      = 'design/head/default_description';
    //////////////////////////

    const ORDER_STATUS_PENDING_CODE = 'kueski_pending';
    const ORDER_STATUS_PENDING_LABEL = 'Pendiente respuesta Kueski';
    const ORDER_STATUS_APPROVED_CODE = 'kueski_approved';
    const ORDER_STATUS_APPROVED_LABEL = 'Kueski aprobado';
    const ORDER_STATUS_FAILED_CODE = 'kueski_failed';
    const ORDER_STATUS_FAILED_LABEL = 'Kueski denegado';

    /**
     * @var KueskiIntegration
     */
    private $integration;

    /**
     * @var ScopeConfigInterface
     */
    protected $_scopeConfig;

    /**
     * @param ScopeConfigInterface $scopeConfig
     * @param KueskiIntegration $integration
     */
    public function __construct(
        ScopeConfigInterface $scopeConfig,
        KueskiIntegration $integration
    ) {
        $this->_scopeConfig = $scopeConfig;
        $this->integration = $integration;
    }

    /**
     * @return bool
     */
    public function isEnabled()
    {
        return $this->_scopeConfig->isSetFlag(
            self::XML_PATH_ENABLE,
            ScopeInterface::SCOPE_STORE
        );
    }

    /**
     * @param string $path
     * @return string
     */
    public function getConfig($path)
    {
        return $this->_scopeConfig->getValue(
            $path,
            ScopeInterface::SCOPE_STORE
        );
    }

    /**
     * @return bool
     */
    public function isSandBox()
    {
        $isSandBox = $this->_scopeConfig->isSetFlag(
            self::XML_SANDBOX,
            ScopeInterface::SCOPE_STORE
        );
        return (bool) $isSandBox;
    }

    /**
     * @return array
     * @throws KueskiException
     */
    public function getMerchantData()
    {
        $data = [];
        $credentials['apiKey']      = $this->getConfig(self::XML_API_KEY);
        $credentials['apiSecret']   = $this->getConfig(self::XML_API_SECRET);
        $credentials['isSandbox']   = $this->isSandBox();

        if (!$credentials['apiKey'] || !$credentials['apiSecret']) {
            throw new KueskiException(__('API Credentials not set'));
        }

        if (!isset($credentials['isSandbox'])) {
            throw new KueskiException(__('Not able to define execution mode'));
        }

        $credentials['isSandbox'] = (bool) $credentials['isSandbox'];
        $merchant = $this->integration->validateCredentials($credentials['apiKey'], $credentials['isSandbox']);

        if (!$merchant) {
            throw new KueskiException(__('Invalid API Credentials'));
        }

        $data = $merchant['data'];
        $data['credentials'] = $credentials;

        return $data;
    }
}
