<?php


namespace Kueski\Payment\Model;


use Kueski\Payment\Model\Payment\Kueski as KueskiPayment;
use Magento\Checkout\Model\ConfigProviderInterface;
use Magento\Framework\App\RequestInterface;
use Magento\Framework\Escaper;
use Magento\Framework\Exception\LocalizedException;
use Magento\Framework\UrlInterface;
use Magento\Framework\View\Asset\Repository;
use Magento\OfflinePayments\Model\Checkmo;
use Magento\Payment\Helper\Data as PaymentHelper;
use Psr\Log\LoggerInterface;

class KueskiConfigProvider implements ConfigProviderInterface
{
    const CHECKOUT_TOTALS_INFORMATION = 'kueski_checkout_totals_information';

    /**
     * @var string[]
     */
    protected $_methodCode = KueskiPayment::PAYMENT_METHOD_KUESKI_CODE;

    /**
     * @var Checkmo
     */
    protected $_method;

    /**
     * @var Escaper
     */
    protected $_escaper;
    /**
     * @var Repository
     */
    protected $_assetRepo;
    /**
     * @var RequestInterface
     */
    protected $_request;
    /**
     * @var \Kueski\Payment\Logger\Logger
     */
    protected $_logger;
    /**
     * @var UrlInterface
     */
    protected $_urlBuilder;

    /**
     * @param PaymentHelper $paymentHelper
     * @param Escaper $escaper
     * @param Repository $assetRepo
     * @param RequestInterface $request
     * @param \Kueski\Payment\Logger\Logger $logger
     * @param UrlInterface $urlBuilder
     * @throws LocalizedException
     */
    public function __construct(
        PaymentHelper                 $paymentHelper,
        Escaper                       $escaper,
        Repository                    $assetRepo,
        RequestInterface              $request,
        \Kueski\Payment\Logger\Logger $logger,
        UrlInterface                  $urlBuilder
    )
    {
        $this->_escaper = $escaper;
        $this->_method = $paymentHelper->getMethodInstance($this->_methodCode);
        $this->_assetRepo = $assetRepo;
        $this->_request = $request;
        $this->_logger = $logger;
        $this->_urlBuilder = $urlBuilder;
    }

    /**
     * {@inheritdoc}
     */
    public function getConfig()
    {
        return array(
            'payment' => array(
                'kueski_payment' => array(
                    'image' => $this->getViewFileUrl("Kueski_Payment::images/LOGO_KUESKIPAY_PRINCIPAL.svg"),
                    'instructions' => $this->_escaper->escapeHtml("<br/>"),
                ),
            ),
        );
    }

    /**
     * Get mailing address from config
     *
     * @return string
     */
    protected function getImageUrl()
    {
        return $this->_method->getPayableTo();
    }

    /**
     * Retrieve url of a view file
     *
     * @param string $fileId
     * @param array $params
     * @return string
     */
    protected function getViewFileUrl($fileId, array $params = array())
    {
        try {
            $params = array_merge(array('_secure' => $this->_request->isSecure()), $params);
            return $this->_assetRepo->getUrlWithParams($fileId, $params);
        } catch (LocalizedException $e) {
            $this->_logger->critical($e->getMessage());
            return $this->_urlBuilder->getUrl('', array('_direct' => 'core/index/notFound'));
        }
    }
}
