<?php
namespace Kueski\Payment\Model\Payment;

use Detection\MobileDetect;
use Exception;
use Kueski\Kueski as KueskiLib;
use Kueski\Payment\Exceptions\KueskiException;
use Magento\Directory\Helper\Data as DirectoryHelper;
use Magento\Framework\Api\AttributeValueFactory;
use Magento\Framework\Api\ExtensionAttributesFactory;
use Magento\Framework\App\Config\ScopeConfigInterface;
use Magento\Framework\App\ProductMetadataInterface;
use Magento\Framework\Controller\ResultFactory;
use Magento\Framework\Data\Collection\AbstractDb;
use Magento\Framework\DataObject;
use Magento\Framework\Exception\LocalizedException;
use Magento\Framework\Message\ManagerInterface;
use Magento\Framework\Model\Context;
use Magento\Framework\Model\ResourceModel\AbstractResource;
use Magento\Framework\Registry;
use Magento\Payment\Helper\Data;
use Magento\Payment\Model\Method\Logger;
use Magento\Quote\Api\Data\CartInterface;
use Magento\Quote\Model\Quote;
use Magento\Sales\Api\Data\OrderItemInterface;
use Magento\Sales\Model\Order;
use Magento\Sales\Model\Order\Address;
use Magento\Sales\Model\Order\Item;

use Kueski\Payment\Model\Config as ConfigModel;


class Kueski extends \Magento\Payment\Model\Method\AbstractMethod
{
    const PAYMENT_METHOD_KUESKI_CODE = 'kueski_payment';

    /**
     * @var string
     */
    protected $_code = self::PAYMENT_METHOD_KUESKI_CODE;

    /**
     * @var bool
     */
    protected $_isOffline = false;
    /**
     * @var bool
     */
    protected $_isGateway = true;

    /**
     * @var bool
     */
    protected $_canOrder = true;

    /**
     * @var bool
     */
    protected $_isInitializeNeeded = true;

    /**
     * @var \Magento\Store\Model\StoreManagerInterface
     */
    protected $_storeManager;

    /**
     * @var
     */
    protected $_kueski;

    /**
     * @var \Kueski\Payment\Logger\Logger
     */
    protected $_kueskiLog;


    /**
     * @var \Kueski\Payment\Helper\Data
     */
    protected $_helper;

    /**
     * @description merchant related data and authorization
     */
    private $merchantAuthorized     = false;
    private $merchantData           = [];
    /**
     * @var ManagerInterface
     */
    private $messageManager;
    /**
     * @var ResultFactory
     */
    private $result;

    /**
     * @param ConfigModel $configModel
     * @param ResultFactory $result
     * @param ManagerInterface $messageManager
     * @param Context $context
     * @param Registry $registry
     * @param ExtensionAttributesFactory $extensionFactory
     * @param AttributeValueFactory $customAttributeFactory
     * @param Data $paymentData
     * @param ScopeConfigInterface $scopeConfig
     * @param Logger $logger
     * @param \Magento\Store\Model\StoreManagerInterface $storeManager
     * @param \Kueski\Payment\Logger\Logger $kueskiLog
     * @param \Kueski\Payment\Helper\Data $helper
     * @param AbstractResource|null $resource
     * @param AbstractDb|null $resourceCollection
     * @param DirectoryHelper|null $directory
     * @param array $data
     */
    public function __construct(
        ConfigModel $configModel,
        ResultFactory $result,
        ManagerInterface $messageManager,
        \Magento\Framework\Model\Context $context,
        \Magento\Framework\Registry $registry,
        \Magento\Framework\Api\ExtensionAttributesFactory $extensionFactory,
        \Magento\Framework\Api\AttributeValueFactory $customAttributeFactory,
        \Magento\Payment\Helper\Data $paymentData,
        \Magento\Framework\App\Config\ScopeConfigInterface $scopeConfig,
        Logger $logger,
        \Magento\Store\Model\StoreManagerInterface $storeManager,
        \Kueski\Payment\Logger\Logger $kueskiLog,
        \Kueski\Payment\Helper\Data $helper,
        \Magento\Framework\Model\ResourceModel\AbstractResource $resource = null,
        \Magento\Framework\Data\Collection\AbstractDb $resourceCollection = null,
        DirectoryHelper $directory = null,
        array $data = array()
    ) {
        parent::__construct($context, $registry, $extensionFactory, $customAttributeFactory, $paymentData, $scopeConfig, $logger, $resource, $resourceCollection, $data, $directory);
        $this->_storeManager        = $storeManager;
        $this->_kueskiLog           = $kueskiLog;
        $this->_helper              = $helper;
        $this->messageManager       = $messageManager;
        $this->configModel          = $configModel;
        $this->result               = $result;
    }

    /**
     * @param $storeId
     * @return bool
     */
    public function isActive($storeId = null)
    {
        return $this->configModel->isEnabled();
    }

    /**
     * @param CartInterface|null $quote
     * @return bool
     */
    public function isAvailable($quote = null) {
        try {
            $isEnabled = $this->configModel->isEnabled();

            $this->merchantData = $this->configModel->getMerchantData();
            $merchantLimits = $this->merchantData['merchant_limits'];

            if (!$isEnabled || $quote == null || empty($merchantLimits)) {
                throw new KueskiException(__('isAvailable: Module disabled or limits not established'));
            }

            if (($quote->getGrandTotal() < $merchantLimits["min_amount"])
                || ($quote->getGrandTotal() > $merchantLimits["max_amount"])){
                throw new KueskiException(__('isAvailable: Order total not meets merchant limits'));
            }
        } catch (KueskiException $e) {
            //Redirect to base cart and show error message
            $this->_kueskiLog->debug($e->getMessage());
            return false;
        }

        return true;
    }

    /**
     * @return string
     */
    public function getConfigPaymentAction()
    {
        return self::ACTION_ORDER;
    }

    /**
     * Instantiate state and set it to state object
     *
     * @param string $paymentAction
     * @param DataObject $stateObject
     * @return Kueski
     * @throws LocalizedException
     */
    public function initialize($paymentAction, $stateObject)
    {
        $stateObject->setState(Order::STATE_PAYMENT_REVIEW);
        $stateObject->setStatus(ConfigModel::ORDER_STATUS_PENDING_CODE);
        $stateObject->setIsNotified(false);

        /** @var \Magento\Sales\Model\Order $order */
        $order = $this->getInfoInstance()->getOrder();
        $baseUrl = $order->getStore()->getBaseUrl();

        if (count($this->merchantData) == 0) {
            $this->merchantData = $this->configModel->getMerchantData();
        }

        $credentials = $this->merchantData['credentials'];
        $successPage = $baseUrl . "kueski/onepage/success/id/" .  $order->getIncrementId();

        $this->_kueskiLog->debug("orderId = ".$order->getId());
        $this->_kueskiLog->debug("orderId = ".$order->getEntityId());


        $this->_kueskiLog->debug(var_export($credentials['apiKey'], true));
        $this->_kueskiLog->debug(var_export($successPage, true));
        $kpSource = $this->isMobile() ? "mobile" : "web";

        try {
            $kueski = new KueskiLib(
                $order->getIncrementId(),
                "M2",
                $credentials['apiKey'],
                $successPage,
                $baseUrl . "kueski/reject/index/id/" . $order->getIncrementId(),
                $baseUrl . "kueski/cancel/index/id/" . $order->getIncrementId(),
                $baseUrl . "kueski/failed/index/id/" . $order->getIncrementId(),
                $this->_helper->getKpName(),
                "v2",
                $kpSource,
                "checkout_button",
                $credentials['isSandbox']
            );

            /** @var Address $shippingAddress */
            $shippingAddress = $order->getShippingAddress();

            /** @var Address $billingAddress */
            $billingAddress = $order->getBillingAddress();

            $colonia = ".";
            if (isset($billingAddress->getStreet()[1])) {
                $colonia = $billingAddress->getStreet()[1];
            }

            $kueski->setBillingAddress(
                $billingAddress->getName(),
                "XAXX010101000", // cambiar si el rfc está en algún lado guardado
                $billingAddress->getStreet()[0],
                "1", // cambiar si el en las direcciones se solicita el interior
                $colonia,
                $billingAddress->getCity(),
                $billingAddress->getRegion(),
                $billingAddress->getPostcode(),
                $billingAddress->getCountryId(),
                $billingAddress->getTelephone(),
                $billingAddress->getEmail()
            );

            $colonia = ".";

            if (!empty($shippingAddress)){
                if (!empty($shippingAddress->getStreet()[1])){
                    $colonia = $shippingAddress->getStreet()[1];
                }
                $kueski->setShippingAddress(
                    $shippingAddress->getFirstname(),
                    $shippingAddress->getLastname(),
                    $shippingAddress->getStreet()[0],
                    "1", // cambiar si el en las direcciones se solicita el interior
                    $colonia,
                    $shippingAddress->getCity(),
                    $shippingAddress->getRegion(),
                    $shippingAddress->getPostcode(),
                    $shippingAddress->getCountryId(),
                    $shippingAddress->getTelephone(),
                    $shippingAddress->getEmail(),
                    "HOME"
                );
            }else{
                $this->_kueskiLog->debug(var_export("USUARIO SIN DIRECCION DE ENVIO", true));
                if (!empty($billingAddress->getStreet()[1])){
                    $colonia = $billingAddress->getStreet()[1];
                }
                $kueski->setShippingAddress(
                    $billingAddress->getFirstname(),
                    $billingAddress->getLastname(),
                    $billingAddress->getStreet()[0],
                    "1", // cambiar si el en las direcciones se solicita el interior
                    $colonia,
                    $billingAddress->getCity(),
                    $billingAddress->getRegion(),
                    $billingAddress->getPostcode(),
                    $billingAddress->getCountryId(),
                    $billingAddress->getTelephone(),
                    $billingAddress->getEmail(),
                    "HOME"
                );
            }

            $needToUseHandlingFee = false;
            $discountPerItem = $order->getDiscountAmount() / $order->getTotalQtyOrdered();

            /** @var Item $item */
            foreach ($order->getAllVisibleItems() as $item) {
                if ($item->getParentItem() instanceof OrderItemInterface) {
                    continue;
                }
                $taxItem =  $item->getRowTotalInclTax() - $item->getRowTotal();
                $totalItem = (($item->getRowTotalInclTax() - $taxItem - $discountPerItem) / $item->getQtyOrdered());
                if ($this->hasMoreThanTwoDecimals($totalItem)){
                    $this->_kueskiLog->debug("needToUseHandlingFee");
                    $this->_kueskiLog->debug("\$discountAmount = [$totalItem]");
                    $needToUseHandlingFee =  true;
                    break;
                }
            }

            /** @var Quote $quote */
            $discountAmount = $order->getDiscountAmount();
            if ($discountAmount < 0) {
                $discountAmount = $discountAmount * (-1);
            }
            $this->_kueskiLog->debug("NeedToUseHandlingFee  = [$needToUseHandlingFee]");
            $this->_kueskiLog->debug("\$discountAmount = [$discountAmount]");
            if ($needToUseHandlingFee){
                $subtotal = 0;
                $subtotal2 = 0 ;
                /** @var Item $item */
                foreach ($order->getAllVisibleItems() as $item) {
                    if ($item->getParentItem() instanceof OrderItemInterface) {
                        continue;
                    }
                    $taxItem =  $item->getRowTotalInclTax() - $item->getRowTotal();
                    $totalItem = (($item->getRowTotalInclTax() - $taxItem - $item->getDiscountAmount()) / $item->getQtyOrdered());
                    $subtotal += $totalItem;
                    $subtotal2 += (bcdiv($totalItem, '1', 2) * $item->getQtyOrdered());
                    $product = $item->getProduct();

                    $kueski->addNewItem(
                        $product->getName(),
                        $product->getName(),
                        $product->getSku(),
                        $item->getQtyOrdered(),
                        (float)bcdiv($totalItem, '1', 2),
                        "MXN",
                        $taxItem
                    );
                    $this->_kueskiLog->debug($product->getSku()." \$totalItem = [" . bcdiv($totalItem, '1', 2) . "]");
                }

                $handlingFee = $order->getSubtotal() - $discountAmount - $subtotal2;

                $kueski->setAmountDetails(
                    $subtotal2,
                    $order->getShippingAmount(),
                    (float)bcdiv($handlingFee, '1', 2),
                    $order->getTaxAmount(),
                    "MXN"
                );

                $this->_kueskiLog->debug("\$subtotal = [" . $subtotal . "]");
                $this->_kueskiLog->debug("\$subtotal2 = [" . $order->getBaseSubtotal() . "]");
                $this->_kueskiLog->debug("\$subtotal3 = [" . $order->getSubtotal() . "]");
                $this->_kueskiLog->debug("\$subtotal4 = [" . $subtotal2 . "]");
                $this->_kueskiLog->debug("\handlingfee = [" . ($order->getSubtotal() - $discountAmount - $subtotal) . "]");
                $this->_kueskiLog->debug("\handlingfee2 = [" . ($order->getSubtotal() - $discountAmount - $subtotal2) . "]");
                $this->_kueskiLog->debug("\$isubtotal2 = [" . ($order->getSubtotal() - $discountAmount) . "]");
                $this->_kueskiLog->debug("\$discountPerItem = [$discountPerItem]");
            }else{
                $this->_kueskiLog->debug("NeedToUseHandlingFee  = [$needToUseHandlingFee]");
                $kueski->setAmountDetails(
                    $order->getSubtotal() - $discountAmount,
                    $order->getShippingAmount(),
                    0.00,
                    $order->getTaxAmount(),
                    "MXN"
                );

                $subtotal = 0;
                /** @var Item $item */
                foreach ($order->getAllVisibleItems() as $item) {
                    if ($item->getParentItem() instanceof OrderItemInterface) {
                        continue;
                    }

                    $taxItem = $item->getRowTotalInclTax() - $item->getRowTotal();
                    $totalItem = $item->getRowTotal() / $item->getQtyOrdered() + $discountPerItem;

                    $this->_kueskiLog->debug("\RowTotal = ".$item->getRowTotal());
                    $this->_kueskiLog->debug("\RowTotal = ".$item->getRowTotalInclTax());
                    $this->_kueskiLog->debug( $item->getRowTotal() / $item->getQtyOrdered());
                    $this->_kueskiLog->debug( $item->getRowTotal() / $item->getQtyOrdered() + $discountPerItem);
                    $this->_kueskiLog->debug("\$totalItem = [$totalItem]");

                    $subtotal += $totalItem;
                    $product = $item->getProduct();
                    $kueski->addNewItem(
                        $product->getName(),
                        $product->getName(),
                        $product->getSku(),
                        $item->getQtyOrdered(),
                        $totalItem,
                        "MXN",
                        $item->getTaxAmount()
                    );
                }

                $this->_kueskiLog->debug("\$discountAmount = [$discountAmount]");
                $this->_kueskiLog->debug("\$discountPerItem = [$discountPerItem]");
                $this->_kueskiLog->debug("\$isubtotal2 = [" . ($order->getSubtotal() - $discountAmount) . "]");
                $this->_kueskiLog->debug("\$subtotal = [$subtotal]");
            }

            $this->_kueskiLog->debug(var_export($kueski->makeJSONArray(), true));
            $this->_kueskiLog->debug("\$apiKey = [" . $credentials['apiKey'] . "]");
            $this->_kueskiLog->debug("\$isSandbox = [" . $credentials['isSandbox'] . "]");

            $result = $kueski->sendOrder();
            $this->_kueskiLog->debug("URL: $result");
            $kueskiId = explode("payment_id=", $result);

            if (is_array($kueskiId)) {
                $order->getPayment()->setAdditionalInformation("Kueski ID", $kueskiId[1]);
                $order->getPayment()->setAdditionalInformation("Kueski URL", $result);
                $this->_kueskiLog->debug("Kueski ID: " . $kueskiId[1]);
                $this->getInfoInstance()->setAdditionalInformation('kueski_authorized_url', $result);
            } else {
                throw new LocalizedException(__($result));
            }
        } catch (Exception $e) {
            $this->_kueskiLog->debug($e->getMessage());
            throw new KueskiException(__($e->getMessage()));
        }

        $order->setStatus(ConfigModel::ORDER_STATUS_PENDING_CODE)
            ->setCanSendNewEmailFlag(false)
            ->setState(Order::STATE_PAYMENT_REVIEW)
            ->save();

        return $this;
    }

    /**
     * @param string $number
     * @return bool
     */
    private function hasMoreThanTwoDecimals($number)
    {
        $number = abs($number);
        $intPart = floor($number);
        $floatPart = $number - $intPart;

        return (strlen($floatPart) > 4);
    }

    /**
     * @return bool
     */
    public function isMobile()
    {
        $detect = new MobileDetect();

        return $detect->isMobile();
    }

    /**
     * @param Order $order
     * @return void
     * @throws LocalizedException
     */
    public function processPayment($order)
    {
        $order->setState(Order::STATE_PROCESSING)
            ->setStatus(ConfigModel::ORDER_STATUS_APPROVED_CODE)
            ->save();
    }

    /**
     * Returns URL to redirect to Payment Gateway
     * @return string | false
     */
    public function redirectGateway()
    {
        return $this->getInfoInstance()->getAdditionalInformation('kueski_authorized_url');
    }

    /**
     * @param $order
     * @return $this
     */
    public function startProcessing($order)
    {
        $order->setState(Order::STATE_PROCESSING)
            ->save();

        return $this;
    }
}
