<?php
namespace Kueski\Payment\Plugin\App\Request;

use Magento\Framework\App\ActionInterface;
use Magento\Framework\App\Area;
use Magento\Framework\App\CsrfAwareActionInterface;
use Magento\Framework\App\Request\CsrfValidator as MagentoCsrfValidator;
use Magento\Framework\App\Request\Http as HttpRequest;
use Magento\Framework\App\Request\InvalidRequestException;
use Magento\Framework\App\RequestInterface;
use Magento\Framework\App\State as AppState;
use Magento\Framework\Controller\Result\RedirectFactory;
use Magento\Framework\Data\Form\FormKey\Validator as FormKeyValidator;
use Magento\Framework\Exception\LocalizedException;
use Magento\Framework\Phrase;

class CsrfValidator
{
    /**
     * @var FormKeyValidator
     */
    protected $_formKeyValidator;

    /**
     * @var RedirectFactory
     */
    protected $_redirectFactory;

    /**
     * @var AppState
     */
    protected $_appState;

    /**
     * @param FormKeyValidator $formKeyValidator
     * @param RedirectFactory $redirectFactory
     * @param AppState $appState
     */
    public function __construct(
        FormKeyValidator $formKeyValidator,
        RedirectFactory $redirectFactory,
        AppState $appState
    ) {
        $this->_formKeyValidator = $formKeyValidator;
        $this->_redirectFactory = $redirectFactory;
        $this->_appState = $appState;
    }

    /**
     * Validate given request.
     *
     * @param HttpRequest $request
     * @param ActionInterface $action
     * @return bool
     */
    public function validateRequest($request, $action) {
        $valid = null;
        if ($action instanceof CsrfAwareActionInterface) {
            $valid = $action->validateForCsrf($request);
        }

        if ($valid === null) {
            $valid = $request->isPost();
        }

        return $valid;
    }

    /**
     * @param HttpRequest $request
     * @param ActionInterface $action
     * @return InvalidRequestException
     */
    public function createException($request, $action) {
        $exception = null;
        if ($action instanceof CsrfAwareActionInterface) {
            $exception = $action->createCsrfValidationException($request);
        }

        if (!$exception) {
            $response = $this->_redirectFactory->create()
                ->setRefererOrBaseUrl()
                ->setHttpResponseCode(302);
            $messages = array(
                new Phrase('Invalid Form Key. Please refresh the page.'),
            );
            $exception = new InvalidRequestException($response, $messages);
        }

        return $exception;
    }

    /**
     * @param MagentoCsrfValidator $validator
     * @param callable $proceed
     * @param RequestInterface $request
     * @param ActionInterface $action
     * @return void
     * @throws InvalidRequestException
     */
    public function aroundValidate(
        $validator,
        callable $proceed,
        $request,
        $action
    ) {
        if ($action instanceof \Kueski\Payment\Controller\Index\Index ) {
            try {
                $areaCode = $this->_appState->getAreaCode();
            } catch (LocalizedException $exception) {
                $areaCode = null;
            }

            if ($request instanceof HttpRequest
                && in_array(
                    $areaCode,
                    array(Area::AREA_FRONTEND, Area::AREA_ADMINHTML),
                    true
                )
            ) {
                $valid = $this->validateRequest($request, $action);
                if (!$valid) {
                    // @codingStandardsIgnoreLine
                    throw $this->createException($request, $action);
                }
            }
        } else {
            $proceed($request, $action);
        }
    }
}
