<?php
declare(strict_types=1);
/**
 * @copyright: Copyright © 2017 Firebear Studio. All rights reserved.
 * @author   : Firebear Studio <fbeardev@gmail.com>
 */

namespace Firebear\ImportExport\Cron;

use Exception;
use Firebear\ImportExport\Model\Job;
use Firebear\ImportExport\Model\Job\Processor;
use Firebear\ImportExport\Model\JobRegistry;
use Firebear\ImportExport\Model\Source\Type\AbstractType;
use Firebear\ImportExport\Model\Source\Type\SearchSourceTypeInterface;
use Firebear\ImportExport\Model\Source\Type\SourceTypeInterface;
use Magento\Framework\Exception\FileSystemException;
use Magento\Framework\Exception\LocalizedException;
use Symfony\Component\Console\Output\OutputInterface;
use Firebear\ImportExport\Model\Job\Handler\HandlerPoolInterface;
use Firebear\ImportExport\Model\Job\Handler\HandlerInterface;
use Firebear\ImportExport\Helper\Data as Helper;

/**
 * Sales entity grids indexing observer.
 *
 * Performs handling cron jobs related to indexing
 * of Order, Invoice, Shipment and Creditmemo grids.
 */
class RunImportJobs
{
    /**
     * @var Processor
     */
    protected $processor;

    /**
     * @var Helper
     */
    protected $helper;

    /**
     * @var bool
     */
    protected $debugMode;

    /**
     * @var HandlerPoolInterface
     */
    private $handlerPool;

    /**
     * @var JobRegistry
     */
    protected $jobRegistry;

    /**
     * @var AbstractType|SearchSourceTypeInterface|SourceTypeInterface
     */
    protected $importSource;

    /**
     * RunImportJobs constructor.
     *
     * @param Processor $importProcessor
     * @param Helper $helper
     * @param HandlerPoolInterface $handlerPool
     * @param JobRegistry $jobRegistry
     */
    public function __construct(
        Processor $importProcessor,
        Helper $helper,
        HandlerPoolInterface $handlerPool,
        JobRegistry $jobRegistry
    ) {
        $this->helper = $helper;
        $this->processor = $importProcessor;
        $this->handlerPool = $handlerPool;
        $this->jobRegistry = $jobRegistry;
    }

    /**
     * @param $schedule
     *
     * @return bool
     * @throws \Magento\Framework\Exception\LocalizedException
     */
    public function execute($schedule)
    {
        $output = null;
        $jobCode = $schedule->getJobCode();

        preg_match('/_id_([0-9]+)/', $jobCode, $matches);
        if (isset($matches[1]) && (int)$matches[1] > 0) {
            $noProblems = 0;
            $jobId = (int)$matches[1];
            $result = false;
            $file = $this->helper->beforeRun($jobId);
            $history = $this->helper->createHistory($jobId, $file, 'console');
            $this->processor->debugMode = $this->debugMode = $this->helper->getDebugMode();
            $this->processor->inConsole = 1;
            $this->processor->setLogger($this->helper->getLogger());
            $job = $this->getJobRepo()->retrieve($jobId);
            $sourceData = $job->getSourceData();
            $importSourceType = $this->getImportSource($sourceData);
            if (isset($sourceData[SearchSourceTypeInterface::SCAN_DIR])
                && $sourceData[SearchSourceTypeInterface::SCAN_DIR] == 1
                && $importSourceType->isSearchable()
            ) {
                try {
                    $this->addLogComment(__('Scan Directory for files'), $output, 'info');
                    $filePath = $importSourceType->getImportFilePath() ?: $sourceData['file_path'] ?? '';
                    $fileType = $sourceData['type_file'];

                    if (!$importSourceType->isExists($filePath)) {
                        throw new LocalizedException(__('Path not exists'));
                    }
                    if (!$importSourceType->isAllowablePath($filePath)) {
                        throw new LocalizedException(__(
                            'Path not allowable. Only %1',
                            implode(', ', $importSourceType->allowedDirsForScan())
                        ));
                    }

                    $files = $importSourceType
                        ->search("$filePath/*.$fileType");
                    if (!empty($files)) {
                        $files = $importSourceType
                            ->filterSearchedFiles($files);
                        foreach ($files as $importFile) {
                            $importFile = $importSourceType
                                ->getFilePath($importFile);
                            $file = $this->helper->beforeRun($jobId);
                            $file .= '-' . basename($importFile);
                            $history = $this->helper->createHistory($jobId, $file, 'console');
                            $this->addLogComment(__('Import File %1', $importFile), $output, 'info');
                            try {
                                $this->processor->setImportFile($importFile);
                                [$noProblems, $result] = $this->processImport($jobId, $file, $history);
                                if (!$this->processor->getIsValidated()) {
                                    $noProblems = 1;
                                }
                            } catch (Exception $exception) {
                                $this->addLogComment(
                                    $exception->getMessage(),
                                    $output,
                                    'error'
                                );
                                $noProblems = 1;
                            }
                            $f = pathinfo($importFile);
                            if ($noProblems) {
                                if (strpos($this->processor->getErrorMessage(), 'Connection Lost') !== false) {
                                    $this->addLogComment(
                                        __('Please re-run the job due to connection lost'),
                                        $output,
                                        'error'
                                    );
                                    return false;
                                }
                                $path = $importSourceType
                                        ->getConfigFilePath() . $importSourceType
                                        ->getDirName(SearchSourceTypeInterface::ERR_DIR);
                            } else {
                                $path = $importSourceType
                                        ->getConfigFilePath() . $importSourceType->getDirName();
                            }
                            $this->addLogComment(
                                __('Moving file to %1 after import', $path . $f['basename']),
                                $output,
                                'info'
                            );
                            $importSourceType->move($importFile, $path . $f['basename']);
                            $this->postImportProcess($job, $file, $result);
                        }
                    }
                } catch (LocalizedException | FileSystemException $localizedException) {
                    $this->addLogComment(
                        __('Error occured detailed error in logger file'),
                        $output,
                        'error'
                    );
                    $this->helper->getLogger()->error($localizedException->getMessage());
                }
            } else {
                try {
                    [$noProblems, $result] = $this->processImport($jobId, $file, $history);
                } catch (LocalizedException $localizedException) {
                    $this->processor->addLogComment(
                        $localizedException->getMessage(),
                        null,
                        'error'
                    );
                }
            }
            if (!$noProblems && $this->processor->reindex) {
                $this->processor->processReindex($file, $jobId);
            }
            $this->postImportProcess($job, $file, $result);
            return true;
        }
        return false;
    }

    /**
     * @return JobRegistry
     */
    private function getJobRepo()
    {
        return $this->jobRegistry;
    }

    /**
     * @param array $sourceData
     * @return AbstractType|SourceTypeInterface|SearchSourceTypeInterface
     * @throws LocalizedException
     */
    protected function getImportSource(array $sourceData)
    {
        if ($this->importSource === null) {
            $this->importSource = $this->processor->getImportModel()
                ->setData($sourceData)
                ->getSource();
        }
        return $this->importSource;
    }

    /**
     * @param $debugData
     * @param OutputInterface|null $output
     * @param null $type
     * @return Processor
     */
    public function addLogComment($debugData, OutputInterface $output = null, $type = null)
    {
        return $this->processor->addLogComment($debugData, $output, $type);
    }

    /**
     * @param int $jobId
     * @param string $file
     * @param $history
     * @return array
     * @throws LocalizedException
     */
    protected function processImport(int $jobId, string $file, $history)
    {
        $noProblems = 0;
        $this->processor->processScope($jobId, $file);
        $counter = $this->helper->countData($file, $jobId);
        $result = $error = 0;
        for ($i = 0; $i < $counter; $i++) {
            list($count, $result) = $this->helper->processImport($file, $jobId, $i, $error, 0);
            $error += $count;
            if (!$result) {
                $noProblems = 1;
                break;
            }
        }
        $this->processor->showErrors();
        $this->processor->getImportModel()->getErrorAggregator()->clear();
        $this->processor->getImportModel()->setNullEntityAdapter();
        $this->helper->saveFinishHistory($history);
        return [$noProblems, $result];
    }

    /**
     * @param Job $job
     * @param string $file
     * @param bool $result
     * @throws LocalizedException
     */
    protected function postImportProcess(Job $job, string $file, bool $result)
    {
        /** @var HandlerInterface $handler */
        foreach ($this->handlerPool->getHandlersInstances() as $handler) {
            $handler->execute($job, $file, (int)$result);
        }
    }
}
