<?php
/**
 * @copyright: Copyright © 2019 Firebear Studio. All rights reserved.
 * @author   : Firebear Studio <fbeardev@gmail.com>
 */

namespace Firebear\ImportExport\Model\Migration\Job;

use Firebear\ImportExport\Model\Migration\AdditionalOptions;
use Symfony\Component\Console\Output\OutputInterface;
use Firebear\ImportExport\Model\Migration\DbConnection;
use Firebear\ImportExport\Model\Migration\Field\Job\MapStoreId;
use Firebear\ImportExport\Model\Migration\FilterJobs\StoreId;
use Firebear\ImportExport\Model\Migration\JobInterface;

/**
 * @inheritdoc
 */
class UrlRewrite implements JobInterface
{
    /**
     * @var DbConnection
     */
    private $dbConnection;

    /**
     * @var StoreId
     */
    private $storeIdFilter;

    /**
     * @var MapStoreId
     */
    private $storeIdMapper;

    /**
     * @param DbConnection $dbConnection
     * @param MapStoreId $storeIdMapper
     * @param StoreId $storeIdFilter
     */
    public function __construct(
        DbConnection $dbConnection,
        StoreId $storeIdFilter,
        MapStoreId $storeIdMapper
    ) {
        $this->dbConnection = $dbConnection;
        $this->storeIdFilter = $storeIdFilter;
        $this->storeIdMapper = $storeIdMapper;
    }

    /**
     * @return DbConnection
     */
    protected function getDbConnection()
    {
        return $this->dbConnection;
    }

    /**
     * @inheritdoc
     */
    public function job($output, $additionalOptions = null)
    {
        $source = $this->getDbConnection()->getSourceChannel();
        $destination = $this->getDbConnection()->getDestinationChannel();

        $sourceSelect = $source->select()
            ->from('core_url_rewrite', [
                'entity_type' => new \Zend_Db_Expr("IF(category_id, 'category', IF(product_id, 'product', null))"),
                'entity_id' => new \Zend_Db_Expr('IF(category_id, category_id, IF(product_id, product_id, null))'),
                'request_path' => 'request_path',
                'target_path' => 'target_path',
                'redirect_type' => new \Zend_Db_Expr("CASE options WHEN 'RP' THEN 301 WHEN 'R' THEN 302 ELSE 0 END"),
                'store_id' => 'store_id',
                'description' => 'description',
                'is_autogenerated' => 'is_system',
                'metadata' => new \Zend_Db_Expr('null'),
            ])->where('is_system = ?', 0);
        $this->storeIdFilter->apply('store_id', $sourceSelect);
        $sourceData = $source->query($sourceSelect);
        $destinationRows = [];
        while ($sourceDataRow = $sourceData->fetch()) {
            $requestPath = $sourceDataRow['request_path'];
            $targetPath = $sourceDataRow['target_path'];
            if (!preg_match('/^.+-(\d+)\/$/', $requestPath)) {
                $existingSelect = $destination->select()
                    ->from('url_rewrite', ['url_rewrite_id'])
                    ->where('request_path = ?', $requestPath)
                    ->where('target_path = ?', $targetPath);

                $existingId = $destination->fetchOne($existingSelect);

                if (!$existingId) {
                    $sourceDataRow['url_rewrite_id'] = null;
                } else {
                    $sourceDataRow['url_rewrite_id'] = $existingId;
                }

                $sourceDataRow['store_id'] = $this->storeIdMapper->job(
                    'store_id',
                    $sourceDataRow['store_id'],
                    'store_id',
                    $sourceDataRow['store_id'],
                    $sourceDataRow
                );

                $destinationRows[] = $sourceDataRow;
            }
        }

        foreach (array_chunk($destinationRows, 1000) as $destinationBatch) {
            $this->getDbConnection()->getDestinationChannel()->insertOnDuplicate(
                'url_rewrite',
                $destinationBatch
            );
        }
    }
}
