<?php
/**
 * @copyright: Copyright © 2020 Firebear Studio. All rights reserved.
 * @author   : Firebear Studio <fbeardev@gmail.com>
 */

namespace Firebear\ImportExport\Model\Migration\Job;

use Firebear\ImportExport\Model\Migration\AdditionalOptions;
use Symfony\Component\Console\Output\OutputInterface;
use Firebear\ImportExport\Model\Migration\DbConnection;
use Firebear\ImportExport\Model\Migration\Field\Job\MapStoreId;
use Firebear\ImportExport\Model\Migration\FilterJobs\StoreId;
use Firebear\ImportExport\Model\Migration\JobInterface;

/**
 * @inheritdoc
 */
class UrlRewriteM2 implements JobInterface
{
    /**
     * @var DbConnection
     */
    private $dbConnection;

    /**
     * @var StoreId
     */
    private $storeIdFilter;

    /**
     * @var MapStoreId
     */
    private $storeIdMapper;

    /**
     * @param DbConnection $dbConnection
     * @param MapStoreId $storeIdMapper
     * @param StoreId $storeIdFilter
     */
    public function __construct(
        DbConnection $dbConnection,
        StoreId $storeIdFilter,
        MapStoreId $storeIdMapper
    ) {
        $this->dbConnection = $dbConnection;
        $this->storeIdFilter = $storeIdFilter;
        $this->storeIdMapper = $storeIdMapper;
    }

    /**
     * @return DbConnection
     */
    protected function getDbConnection()
    {
        return $this->dbConnection;
    }

    /**
     * @inheritdoc
     */
    public function job($output, $additionalOptions = null)
    {
        $source = $this->getDbConnection()->getSourceChannel();
        $destination = $this->getDbConnection()->getDestinationChannel();
        $this->dbConnection->getDestinationChannel()->query('SET FOREIGN_KEY_CHECKS = 0;');
        $this->dbConnection->getDestinationChannel()->truncateTable('url_rewrite');
        $this->dbConnection->getDestinationChannel()->query('SET FOREIGN_KEY_CHECKS = 1;');

        $sourceSelect = $source->select()
            ->from('url_rewrite', [
                'entity_type' => 'entity_type',
                'entity_id' => 'entity_id',
                'request_path' => 'request_path',
                'target_path' => 'target_path',
                'redirect_type' => 'redirect_type',
                'store_id' => 'store_id',
                'description' => 'description',
                'is_autogenerated' => 'is_autogenerated',
                'metadata' => 'metadata',
            ]);
        $this->storeIdFilter->apply('store_id', $sourceSelect);
        $sourceData = $source->query($sourceSelect);
        $destinationRows = [];
        while ($sourceDataRow = $sourceData->fetch()) {
            $requestPath = $sourceDataRow['request_path'];
            $targetPath = $sourceDataRow['target_path'];
            if (!preg_match('/^.+-(\d+)\/$/', $requestPath)) {
                $existingSelect = $destination->select()
                    ->from('url_rewrite', ['url_rewrite_id'])
                    ->where('request_path = ?', $requestPath)
                    ->where('target_path = ?', $targetPath);

                $existingId = $destination->fetchOne($existingSelect);

                if (!$existingId) {
                    $sourceDataRow['url_rewrite_id'] = null;
                } else {
                    $sourceDataRow['url_rewrite_id'] = $existingId;
                }

                $sourceDataRow['store_id'] = $this->storeIdMapper->job(
                    'store_id',
                    $sourceDataRow['store_id'],
                    'store_id',
                    $sourceDataRow['store_id'],
                    $sourceDataRow
                );

                $destinationRows[] = $sourceDataRow;
            }
        }

        foreach (array_chunk($destinationRows, 2000) as $destinationBatch) {
            $this->getDbConnection()->getDestinationChannel()->insertOnDuplicate(
                'url_rewrite',
                $destinationBatch
            );
        }
    }
}
